package babase.ranker;

import java.awt.Component;
import java.util.Vector;

import javax.swing.JOptionPane;

import babase.db.Babase;
import babase.db.RankingInfo;

public class IncorporatorUI {

    /**
     * Incorporate an existing ranking (<code>existingRankingInfo</code>)
     * into user's current ranking (accessible through <code>cage</code>). In
     * general, the two input rankings may involve different sets of
     * individuals. In the proposed new ranking, individuals common to both
     * input rankings will be ranked first according to the existing ranking,
     * followed by individuals unique to the user's ranking, ranked according to
     * the user's ranking.
     * 
     * @param parent
     * @param title
     * @param cage
     * @param existingRankingInfo
     * @param silentIfSameIndividuals
     */
    public static void incorporate(Component parent, String title, Cage cage,
            final RankingInfo existingRankingInfo,
            boolean silentIfSameIndividuals) {
        final Vector<String> referenceSnames = existingRankingInfo.snames;
        final Vector<Integer> referenceRanks = existingRankingInfo.ranks;
        final Vector<String> targetSnames = new Vector<String>(
                cage.individuals.length);
        for (int i = 0; i < cage.individuals.length; i++)
            targetSnames.add(cage.individuals[i].sname);
        final Vector<Integer> targetRanks = new Vector<Integer>(cage.ranks
                .size());
        for (int i = 0; i < cage.ranks.size(); i++)
            targetRanks.add(cage.ranks.get(i));
        // The following array maps an individual index in target to its new
        // consolidated rank:
        final int[] targetIndividualToNewRank = new int[targetSnames.size()];
        for (int i = 0; i < targetIndividualToNewRank.length; i++) {
            targetIndividualToNewRank[i] = -1;
        }
        // Go through the reference ranking in order, and identify those
        // individuals who are common to both reference and target. At the end
        // of the loop, targetIndividualToNewRank should set correct new
        // ranks for these individuals; they will be ranked upfront, and ordered
        // according to references:
        int numCommonIndividuals = 0;
        for (int i = 0; i < referenceRanks.size(); i++) {
            final int targetIndividualIndex = targetSnames
                    .indexOf(referenceSnames.get(referenceRanks.get(i)));
            if (targetIndividualIndex != -1) {
                // This individual is common to both refernce and target:
                targetIndividualToNewRank[targetIndividualIndex] = numCommonIndividuals;
                numCommonIndividuals++;
            }
        }
        // Go through the target ranking in order, and focus on those
        // individuals who are only in target. At the end of the
        // loop, targetIndividualToNewRank should set correct new ranks for
        // these individuals; they will be ordered as in target, but after those
        // individuals who are also in reference:
        int numRemainingTargetIndividuals = 0;
        for (int i = 0; i < targetRanks.size(); i++) {
            final int individualIndex = targetRanks.get(i);
            if (targetIndividualToNewRank[individualIndex] == -1) {
                // This individual in only in target:
                targetIndividualToNewRank[individualIndex] = numCommonIndividuals
                        + numRemainingTargetIndividuals;
                numRemainingTargetIndividuals++;
            }
        }
        assert (numCommonIndividuals + numRemainingTargetIndividuals == targetSnames
                .size());
        // Convert targetIndividualToNewRank to a ranks array that can be used
        // to permute the cage:
        final int[] newRanks = new int[targetRanks.size()];
        for (int i = 0; i < targetIndividualToNewRank.length; i++) {
            newRanks[targetIndividualToNewRank[i]] = i;
        }
        // Now, permute the cage:
        boolean sameIndividuals = (numRemainingTargetIndividuals == 0);
        if (!sameIndividuals || !silentIfSameIndividuals) {
            // Show information:
            final StringBuffer buffer = new StringBuffer();
            buffer.append("<html>");
            buffer.append("Ranking for " + existingRankingInfo.rankType.type
                    + ":" + existingRankingInfo.group.name + " on "
                    + Babase.dateToString(existingRankingInfo.startDate)
                    + " from Babase:<br>");
            for (int i = 0; i < referenceRanks.size(); i++) {
                final String sname = referenceSnames.get(referenceRanks.get(i));
                String output = sname;
                if (targetSnames.contains(sname)) {
                    output = "<b><i>" + output + "</i></b>";
                }
                buffer.append(output);
                buffer.append((i == referenceRanks.size() - 1) ? "<br>" : " ");
            }
            buffer.append("Your ranking:<br>");
            for (int i = 0; i < targetRanks.size(); i++) {
                final String sname = targetSnames.get(targetRanks.get(i));
                String output = sname;
                if (targetIndividualToNewRank[i] < numCommonIndividuals) {
                    output = "<b><i>" + output + "</i></b>";
                }
                buffer.append(output);
                buffer.append((i == targetRanks.size() - 1) ? "<br>" : " ");
            }
            buffer.append("Your ranking will be changed to:<br>");
            for (int i = 0; i < newRanks.length; i++) {
                final String sname = targetSnames.get(newRanks[i]);
                String output = sname;
                if (i < numCommonIndividuals) {
                    output = "<b><i>" + output + "</i></b>";
                }
                buffer.append(output);
                buffer.append((i == newRanks.length - 1) ? "<br>" : " ");
            }
            buffer.append("Do you wish to proceed?");
            buffer.append("</html>");
            final String[] options = { "Yes", "No" };
            final int n = JOptionPane.showOptionDialog(parent, buffer
                    .toString(), title, JOptionPane.YES_NO_OPTION,
                    JOptionPane.WARNING_MESSAGE, null, options, options[0]);
            if (n == 1)
                return;
        }
        cage.ranks.permute(newRanks, null);
        return;
    }

}
