package babase.ranker;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.sql.SQLException;

import info.clearthought.layout.TableLayout;
import info.clearthought.layout.TableLayoutConstraints;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.plaf.FontUIResource;

import babase.db.Babase;
import babase.db.RankingInfo;
import babase.db.Babase.DumpToXMLException;
import babase.ranker.auto.AlphabeticalAutoRanker;
import babase.ranker.auto.DefaultAutoRanker;
import babase.ranker.ranking.RankingChangeEvent;
import babase.ranker.ranking.RankingChangedListener;

public class MainUI extends JFrame {

    /**
     * 
     */
    private static final long serialVersionUID = -1020260228492294181L;

    private final MainUI _this = this;

    private final Babase _db;

    private final Cage _cage;

    private boolean _ranksSaved = false;

    private int _fontSizeDelta = 0;

    private final InteractionMatrixDisplay _interactionMatrixDisplay;

    private final SlidableHeader _slidableHeader;

    private final SwappableHeader _swappableHeader;

    private final StatsPanel _statsPanel;

    private final HistoryPanel _historyPanel;

    private final PrintStream _out;

    private final JFileChooser _xmlFileChooser;

    private final JFileChooser _textFileChooser;

    private final WindowListener _windowListener;

    private final ActionListener _actionListener;

    private final LoginUI _loginUI;

    private final LoadUI _loadUI;

    public void ranksSaved() {
        _ranksSaved = true;
        _cage.ranks.addChangedListener(new RankingChangedListener() {
            public void rankingChanged(RankingChangeEvent e) {
                _ranksSaved = false;
                _cage.ranks.removeChangedListener(this);
                return;
            }
        });
        return;
    }

    private boolean confirmClosing() {
        if (!_ranksSaved) {
            String[] options = { "Yes", "No" };
            int n = JOptionPane.showOptionDialog(this,
                    "This ranking has not been saved.\n"
                            + "Are you sure you want to close it?", "Warning",
                    JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE,
                    null, options, options[1]);
            return (n == 0);
        }
        return true;
    }

    private void saveToDatabase() {
        if (!Babase.isStartOfMonth(_cage.rankingId.startDate)
                || !Babase.isEndOfMonth(_cage.rankingId.endDate)
                || !Babase.isInSameMonth(_cage.rankingId.startDate,
                        _cage.rankingId.endDate)) {
            JOptionPane
                    .showMessageDialog(
                            this,
                            "Cannot save to Babase: ranking does not start and end on\n"
                                    + "the first and the last days of the same month!\n"
                                    + "(You can still save it in XML or print as text.)");
            return;
        }
        try {
            RankingInfo existingRankingInfo = _db
                    .getRankingInfo(_cage.rankingId);
            boolean proceed = true;
            if (existingRankingInfo != null) {
                StringBuffer buf = new StringBuffer();
                buf.append("Existing ranking found for "
                        + _cage.rankingId.toString() + ":\n");
                for (int i = 0; i < existingRankingInfo.snames.size(); i++) {
                    buf.append(existingRankingInfo.snames
                            .get(existingRankingInfo.ranks.get(i)));
                    if (i < existingRankingInfo.snames.size() - 1)
                        buf.append(" ");
                }
                buf.append("\nYour ranking:\n");
                for (int i = 0; i < _cage.individuals.length; i++) {
                    buf.append(_cage.individuals[_cage.ranks.get(i)].sname);
                    if (i < _cage.individuals.length - 1)
                        buf.append(" ");
                }
                buf.append("\nDo you want to overwrite the existing ranking?");
                String[] options = { "Yes", "No" };
                int n = JOptionPane
                        .showOptionDialog(this, buf.toString(),
                                "Saving to Babase", JOptionPane.YES_NO_OPTION,
                                JOptionPane.QUESTION_MESSAGE, null, options,
                                options[1]);
                proceed = (n == 0);
            } else {
                String[] options = { "Yes", "No" };
                int n = JOptionPane
                        .showOptionDialog(
                                this,
                                "Are you sure you want to save it to the database?",
                                "Saving to Babase", JOptionPane.YES_NO_OPTION,
                                JOptionPane.QUESTION_MESSAGE, null, options,
                                options[1]);
                proceed = (n == 0);
            }
            if (proceed) {
                _db.writeRankingInfo(_cage.rankingId, _cage.individuals,
                        _cage.ranks.get());
                ranksSaved();
                Utils.showMessageDialog(this, "Ranking saved to Babase!");
            }
        } catch (SQLException e) {
            Utils.showErrorDialog(this, e.getMessage());
        }
        return;
    }

    private JMenu setupFileDatabaseMenu() {
        JMenu fileDatabaseMenu = new JMenu("File/Database");
        fileDatabaseMenu.setMnemonic(KeyEvent.VK_F);
        // Save Ranking to Babase:
        JMenuItem saveToDatabase = new JMenuItem("Save Ranking to Babase");
        saveToDatabase.setMnemonic(KeyEvent.VK_S);
        saveToDatabase.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (!_db.isConnected()) {
                    _loginUI.start(new ActionListener() {
                        public void actionPerformed(ActionEvent e) {
                            saveToDatabase();
                            return;
                        }
                    }, new ActionListener() {
                        public void actionPerformed(ActionEvent e) {
                            return;
                        }
                    });
                } else {
                    saveToDatabase();
                }
                return;
            }
        });
        fileDatabaseMenu.add(saveToDatabase);
        JMenuItem dumpToFile = new JMenuItem("Dump Ranker State to XML File");
        dumpToFile.setMnemonic(KeyEvent.VK_D);
        dumpToFile.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (_xmlFileChooser.showOpenDialog(_this) == JFileChooser.APPROVE_OPTION) {
                    if (_xmlFileChooser.getSelectedFile().exists()) {
                        String[] options = { "Yes", "No" };
                        int n = JOptionPane.showOptionDialog(_this,
                                "Overwrite the existing file?", "Warning",
                                JOptionPane.YES_NO_OPTION,
                                JOptionPane.QUESTION_MESSAGE, null, options,
                                options[1]);
                        if (n != 0)
                            return;
                    }
                    try {
                        _cage.dumpToXML(_xmlFileChooser.getSelectedFile());
                        ranksSaved();
                        Utils.showMessageDialog(_this,
                                "Ranker state successfully dumped.");
                    } catch (DumpToXMLException exception) {
                        Utils.showErrorDialog(_this, exception.getMessage());
                    }
                }
                return;
            }
        });
        fileDatabaseMenu.add(dumpToFile);
        JMenuItem printRank = new JMenuItem("Print Ranker State to Text File");
        printRank.setMnemonic(KeyEvent.VK_P);
        printRank.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (_textFileChooser.showSaveDialog(_this) == JFileChooser.APPROVE_OPTION) {
                    if (_textFileChooser.getSelectedFile().exists()) {
                        String[] options = { "Yes", "No" };
                        int n = JOptionPane.showOptionDialog(_this,
                                "Overwrite the existing file?", "Warning",
                                JOptionPane.YES_NO_OPTION,
                                JOptionPane.QUESTION_MESSAGE, null, options,
                                options[1]);
                        if (n != 0)
                            return;
                    }
                    PrintStream ps = null;
                    try {
                        ps = new PrintStream(new FileOutputStream(
                                _textFileChooser.getSelectedFile(), false));
                    } catch (FileNotFoundException exception) {
                        Utils.showErrorDialog(_this, "Error printing to file: "
                                + exception.getMessage());
                        return;
                    }
                    _cage.print(ps);
                    _statsPanel.print(ps);
                    ps.close();
                    Utils
                            .showMessageDialog(_this,
                                    "File successfully printed.");
                }
                return;
            }
        });
        fileDatabaseMenu.add(printRank);
        JMenuItem closeRank = new JMenuItem("Close Current Ranking");
        closeRank.setMnemonic(KeyEvent.VK_C);
        closeRank.setActionCommand("main.close");
        closeRank.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (confirmClosing())
                    _actionListener.actionPerformed(e);
                return;
            }
        });
        fileDatabaseMenu.add(closeRank);
        JMenuItem exit = new JMenuItem("Exit");
        exit.setMnemonic(KeyEvent.VK_E);
        exit.setActionCommand("exit");
        exit.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (confirmClosing())
                    _actionListener.actionPerformed(e);
                return;
            }
        });
        fileDatabaseMenu.add(exit);
        return fileDatabaseMenu;
    }

    private void updateUI() {
        setVisible(false);
        float scale = (float) Math.pow(1.1, _fontSizeDelta);
        final Class[] classes = Main.CLASSES_WITH_CHANGABLE_FONT_SIZES;
        for (int i = 0; i < classes.length; i++) {
            final Font font = (Font) UIManager.get(classes[i].getName()
                    + ".font.default");
            UIManager.put(classes[i].getName() + ".font", new FontUIResource(
                    font.deriveFont(font.getSize() * scale)));
        }
        SwingUtilities.updateComponentTreeUI(_this);
        pack();
        setVisible(true);
        return;
    }

    private JMenu setupRankMenu() {
        JMenu rankMenu = new JMenu("Rank");
        rankMenu.setMnemonic(KeyEvent.VK_R);
        // Load/Incorporate Ranking from Babase:
        JMenuItem loadFromDatabase = new JMenuItem(
                "Load/Incorporate from Babase");
        loadFromDatabase.setMnemonic(KeyEvent.VK_L);
        loadFromDatabase.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                final ActionListener loadedActionListener = new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        return;
                    }
                };
                if (!_db.isConnected()) {
                    _loginUI.start(new ActionListener() {
                        public void actionPerformed(ActionEvent e) {
                            _loadUI.start(loadedActionListener);
                            return;
                        }
                    }, new ActionListener() {
                        public void actionPerformed(ActionEvent e) {
                            return;
                        }
                    });
                } else {
                    _loadUI.start(loadedActionListener);
                }
                return;
            }
        });
        rankMenu.add(loadFromDatabase);
        JMenuItem alphabeticalRank = new JMenuItem("Alphabetical");
        alphabeticalRank.setMnemonic(KeyEvent.VK_A);
        alphabeticalRank.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                int[] newRanks = AlphabeticalAutoRanker.rank(_cage);
                _cage.ranks.permute(newRanks, null);
                return;
            }
        });
        rankMenu.add(alphabeticalRank);
        JMenuItem defaultRank = new JMenuItem("Default Rules");
        defaultRank.setMnemonic(KeyEvent.VK_D);
        defaultRank.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                int[] newRanks = DefaultAutoRanker.rank(_cage);
                _cage.ranks.permute(newRanks, null);
                return;
            }
        });
        rankMenu.add(defaultRank);
        return rankMenu;
    }

    private JMenu setupViewMenu() {
        JMenu viewMenu = new JMenu("View");
        viewMenu.setMnemonic(KeyEvent.VK_V);
        JMenuItem decreaseFont = new JMenuItem("Decrease Font Size");
        decreaseFont.setMnemonic(KeyEvent.VK_D);
        decreaseFont.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _fontSizeDelta--;
                updateUI();
                return;
            }
        });
        viewMenu.add(decreaseFont);
        JMenuItem resetFont = new JMenuItem("Reset Font Size");
        resetFont.setMnemonic(KeyEvent.VK_R);
        resetFont.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _fontSizeDelta = 0;
                updateUI();
                return;
            }
        });
        viewMenu.add(resetFont);
        JMenuItem increaseFont = new JMenuItem("Increase Font Size");
        increaseFont.setMnemonic(KeyEvent.VK_I);
        increaseFont.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _fontSizeDelta++;
                updateUI();
                return;
            }
        });
        viewMenu.add(increaseFont);
        return viewMenu;
    }

    private JMenu setupHelpMenu() {
        JMenu helpMenu = new JMenu("Help");
        helpMenu.setMnemonic(KeyEvent.VK_H);
        JMenuItem about = new JMenuItem("About");
        about.setMnemonic(KeyEvent.VK_A);
        about.setActionCommand("about");
        about.addActionListener(_actionListener);
        helpMenu.add(about);
        return helpMenu;
    }

    public MainUI(WindowListener windowListner, ActionListener actionListener,
            Babase db, Cage cage, JFileChooser textFileChooser,
            JFileChooser xmlFileChooser, LoginUI loginUI) {

        setTitle(cage.rankingId.toString() + "; "
                + cage.interactionSpec.toString());
        boolean showRank = true;

        _db = db;
        _cage = cage;
        _textFileChooser = textFileChooser;
        _xmlFileChooser = xmlFileChooser;
        _windowListener = windowListner;
        _actionListener = actionListener;
        _loginUI = loginUI;
        _loadUI = new LoadUI(this, db, cage);

        this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                if (confirmClosing())
                    _windowListener.windowClosing(e);
                return;
            }
        });

        // Create menu bar elements:
        JMenuBar bar = new JMenuBar();
        bar.add(setupFileDatabaseMenu());
        bar.add(setupRankMenu());
        bar.add(setupViewMenu());
        bar.add(setupHelpMenu());
        setJMenuBar(bar);

        // The data panel diplays the interaction matrix, together with
        // individuals that allow ranking to be manipulated by the user:
        JPanel dataPanel = new JPanel();
        _slidableHeader = new SlidableHeader(_cage.ranks, _cage.individuals);
        _swappableHeader = new SwappableHeader(_cage.ranks, _cage.individuals);
        _interactionMatrixDisplay = new InteractionMatrixDisplay(_cage.ranks,
                _cage.matrix);

        int totalWidth = 1 + (showRank ? 1 : 0) + _cage.ranks.size();
        int totalHeight = 1 + (showRank ? 1 : 0) + _cage.ranks.size();
        double[] tableLayoutWidths = new double[totalWidth];
        double[] tableLayoutHeights = new double[totalHeight];

        for (int i = 0; i < totalWidth; i++)
            // upperTableLayoutWidths[i] = TableLayout.FILL;
            tableLayoutWidths[i] = TableLayout.PREFERRED;
        for (int i = 0; i < totalHeight; i++)
            // upperTableLayoutHeights[i] = TableLayout.FILL;
            tableLayoutHeights[i] = TableLayout.PREFERRED;

        tableLayoutWidths[0] = TableLayout.PREFERRED;
        tableLayoutHeights[0] = TableLayout.PREFERRED;
        if (showRank) {
            tableLayoutWidths[1] = TableLayout.PREFERRED;
            tableLayoutHeights[1] = TableLayout.PREFERRED;
        }

        dataPanel.setLayout(new TableLayout(tableLayoutWidths,
                tableLayoutHeights));
        dataPanel.add(_slidableHeader, new TableLayoutConstraints(
                1 + (showRank ? 1 : 0), 0, totalWidth - 1, 0));
        dataPanel.add(_swappableHeader, new TableLayoutConstraints(0,
                1 + (showRank ? 1 : 0), 0, totalHeight - 1));
        dataPanel.add(_interactionMatrixDisplay, new TableLayoutConstraints(
                1 + (showRank ? 1 : 0), 1 + (showRank ? 1 : 0), totalWidth - 1,
                totalHeight - 1));
        if (showRank) {
            for (int i = 0; i < _cage.ranks.size(); i++) {
                dataPanel.add(new RankLabel(i), new TableLayoutConstraints(
                        2 + i, 1));
                dataPanel.add(new RankLabel(i), new TableLayoutConstraints(1,
                        2 + i));
            }
        }
        JScrollPane dataScrollPane = new JScrollPane(dataPanel,
                ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
                ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        // Normally the following is fine, but if dataPanel's preferred
        // size is larger than the screen size, the resulting dataScrollPane
        // size will be squished:
        // dataScrollPane.setPreferredSize(dataPanel.getPreferredSize());
        dataScrollPane.setPreferredSize(new Dimension(Toolkit
                .getDefaultToolkit().getScreenSize().width / 2, Toolkit
                .getDefaultToolkit().getScreenSize().height * 3 / 4));

        // The tabbed pane displays a variety of useful information:
        _statsPanel = new StatsPanel(_interactionMatrixDisplay, _cage.ranks,
                _cage.individuals, _cage.matrix);
        _historyPanel = new HistoryPanel(_cage.ranks);
        JTabbedPane tabbedPane = new JTabbedPane();
        tabbedPane.addTab("Stats", null, _statsPanel,
                "Show stats about the current ranking");
        tabbedPane.addTab("History", null, _historyPanel,
                "Show/undo history of reordering operations");
        tabbedPane.setSelectedIndex(0);
        // Some size tweaks:
        tabbedPane.setMinimumSize(new Dimension(20, 20));
        tabbedPane.setPreferredSize(_statsPanel.getPreferredSize());

        // The message panel diplays feedbacks to the user:
        MessagePanel messagePanel = new MessagePanel(500);

        _out = messagePanel.getPrintStream();
        _slidableHeader.setOut(_out);
        _swappableHeader.setOut(_out);
        _historyPanel.setOut(_out);

        // Lay out the three top-level panels with split panes:
        JSplitPane splitTopPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
                dataScrollPane, tabbedPane);
        JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
                splitTopPane, messagePanel);
        getContentPane().add(splitPane);

        pack();
        setLocationRelativeTo(null);
        setVisible(true);
    }

    /**
     * Implements the display of a rank number (converted to <code>1</code>-based
     * for display) along with the interaction _cage.matrix.
     * 
     * @author junyang
     * 
     */
    public static class RankLabel extends JLabel {

        private static final long serialVersionUID = 1L;

        /**
         * Construct a <code>RankLabel</code> that displays <code>i+1</code>
         * as the rank.
         * 
         * @param i
         *            The <code>0</code>-based rank.
         */
        public RankLabel(int i) {
            super();
            setHorizontalAlignment(JLabel.CENTER);
            setFont((Font) UIManager.get(getClass().getName() + ".font"));
            setOpaque(true);
            setBackground(Color.DARK_GRAY);
            setForeground(Color.LIGHT_GRAY);
            setText(Integer.toString(i + 1));
            setToolTipText(Integer.toString(i + 1));
            return;
        }

        public void updateUI() {
            super.updateUI();
            setFont((Font) UIManager.get(getClass().getName() + ".font"));
            return;
        }
    }

    public void refresh() {
        _interactionMatrixDisplay.refresh();
        _swappableHeader.refresh();
        _slidableHeader.refresh();
        _statsPanel.redisplay();
        return;
    }

}
