package babase.ranker;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowListener;
import java.sql.SQLException;
import java.util.Date;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import com.toedter.calendar.JDateChooser;

import babase.db.Babase;
import babase.db.Group;
import babase.db.Individual;
import babase.db.InteractionSpec;
import babase.db.RankType;
import babase.db.RankingIdentifier;
import babase.db.RankingInfo;
import babase.ranker.auto.DefaultAutoRanker;

public class ChooserUI extends JFrame {

    private static final long serialVersionUID = 1L;

    private JComboBox rankTypeBox;

    private JComboBox groupBox;

    private JDateChooser rankStartDateChooser;

    private JDateChooser rankEndDateChooser;

    private JComboBox interactionClassChooser;

    private JDateChooser interactionStartDateChooser;

    private JDateChooser interactionEndDateChooser;

    public ChooserUI(Babase db, WindowListener windowListener,
            ActionListener actionListener) throws SQLException {

        setTitle("Ranking from database");
        setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        addWindowListener(windowListener);

        rankTypeBox = new JComboBox(db.getRankTypes());
        groupBox = new JComboBox(db.getGroups());
        rankStartDateChooser = new JDateChooser("yyyy-MM-dd", "####-##-##", '_');
        rankEndDateChooser = new JDateChooser("yyyy-MM-dd", "####-##-##", '_');
        interactionClassChooser = new JComboBox(db.getInteractionClasses());
        interactionStartDateChooser = new JDateChooser("yyyy-MM-dd",
                "####-##-##", '_');
        interactionEndDateChooser = new JDateChooser("yyyy-MM-dd",
                "####-##-##", '_');

        JPanel options = new JPanel(new GridBagLayout());
        JPanel info = new JPanel(new GridBagLayout());

        GridBagConstraints c = new GridBagConstraints();

        c.fill = GridBagConstraints.HORIZONTAL;
        c.insets = new Insets(10, 0, 5, 0);
        info.add(new JLabel("Please specify the following information:"), c);

        c.gridwidth = 1;
        c.gridy = 0;
        c.gridx = 0;
        c.insets = new Insets(10, 10, 0, 10);
        options.add(new JLabel("Rank type: "), c);
        c.gridx = 1;
        c.insets = new Insets(10, 0, 0, 10);
        options.add(rankTypeBox, c);
        c.gridy++;
        c.gridx = 0;
        c.insets = new Insets(10, 10, 0, 10);
        options.add(new JLabel("Group: "), c);
        c.gridx = 1;
        c.insets = new Insets(10, 0, 0, 10);
        options.add(groupBox, c);
        c.gridy++;
        c.gridx = 0;
        c.insets = new Insets(10, 10, 0, 0);
        c.gridwidth = 1;
        options.add(new JLabel("Ranking start date: "), c);
        c.gridx = 1;
        c.insets = new Insets(10, 0, 0, 10);
        options.add(rankStartDateChooser, c);
        c.gridy++;
        c.gridx = 0;
        c.insets = new Insets(10, 10, 0, 0);
        options.add(new JLabel("Ranking end date: "), c);
        c.gridx = 1;
        c.insets = new Insets(10, 0, 0, 10);
        options.add(rankEndDateChooser, c);
        c.gridy++;
        c.gridx = 0;
        c.insets = new Insets(10, 10, 0, 0);
        c.gridwidth = 1;
        options.add(new JLabel("Interaction class: "), c);
        c.gridx = 1;
        c.insets = new Insets(10, 0, 0, 10);
        options.add(interactionClassChooser, c);
        c.gridy++;
        c.gridx = 0;
        c.insets = new Insets(10, 10, 0, 0);
        c.gridwidth = 1;
        options.add(new JLabel("Interaction start date: "), c);
        c.gridx = 1;
        c.insets = new Insets(10, 0, 0, 10);
        options.add(interactionStartDateChooser, c);
        c.gridy++;
        c.gridx = 0;
        c.insets = new Insets(10, 10, 0, 0);
        options.add(new JLabel("Interaction end date: "), c);
        c.gridx = 1;
        c.insets = new Insets(10, 0, 0, 10);
        options.add(interactionEndDateChooser, c);
        c.gridy++;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridx = 0;
        c.insets = new Insets(10, 10, 5, 10);
        JButton cancelButton = new JButton("Cancel");
        cancelButton.setActionCommand("chooser.cancel");
        cancelButton.addActionListener(actionListener);
        cancelButton.setMnemonic(KeyEvent.VK_C);
        options.add(cancelButton, c);
        c.gridx = 1;
        c.insets = new Insets(10, 10, 5, 10);
        JButton rankButton = new JButton("Rank");
        rankButton.setActionCommand("chooser.rank");
        rankButton.addActionListener(actionListener);
        rankButton.setMnemonic(KeyEvent.VK_R);
        options.add(rankButton, c);

        getContentPane().setLayout(new BorderLayout());
        getContentPane().add(info, BorderLayout.NORTH);
        getContentPane().add(options, BorderLayout.SOUTH);

        setResizable(false);
        pack();
        setLocationRelativeTo(null);
        getRootPane().setDefaultButton(rankButton);
    }

    public Cage createCage(Babase db) {
        Cage cage = null;
        RankType rankType = (RankType) rankTypeBox.getSelectedItem();
        Group group = (Group) groupBox.getSelectedItem();
        Date rankStartDate = rankStartDateChooser.getDate();
        Date rankEndDate = rankEndDateChooser.getDate();
        String interactionClass = (String) interactionClassChooser
                .getSelectedItem();
        Date interactionStartDate = interactionStartDateChooser.getDate();
        Date interactionEndDate = interactionEndDateChooser.getDate();
        if (rankStartDate == null || rankEndDate == null
                || !rankStartDate.before(rankEndDate)) {
            JOptionPane.showMessageDialog(this,
                    "Please enter a valid date range for the ranking!");
            return null;
        }
        if (!Babase.isStartOfMonth(rankStartDate)
                || !Babase.isEndOfMonth(rankEndDate)
                || !Babase.isInSameMonth(rankStartDate, rankEndDate)) {
            String[] options = { "Proceed", "Cancel" };
            final int n = JOptionPane
                    .showOptionDialog(
                            this,
                            "Ranking does not start and end on the first and the last days\n"
                                    + "of the same month; you will NOT be able to save it in Babase\n"
                                    + "(but it still can be saved as XML or printed as text).",
                            null, JOptionPane.YES_NO_OPTION,
                            JOptionPane.WARNING_MESSAGE, null, options,
                            options[0]);
            if (n == 1)
                return null;
        }
        if (interactionStartDate == null || interactionEndDate == null
                || !interactionStartDate.before(interactionEndDate)) {
            JOptionPane
                    .showMessageDialog(this,
                            "Please enter a valid date range for the interaction matrix!");
            return null;
        }
        try {
            RankingIdentifier rankingIdentifier = new RankingIdentifier(group,
                    rankType, rankStartDate, rankEndDate);
            InteractionSpec interactionSpec = new InteractionSpec(
                    interactionClass, interactionStartDate, interactionEndDate);
            Individual[] individuals = db.getIndividuals(rankingIdentifier);
            if (individuals.length < 1) {
                Utils
                        .showMessageDialog(this,
                                "There are no individuals to rank between the ranking start and end dates!");
                return null;
            }
            int[][] matrix = db.getInteractionMatrix(rankingIdentifier,
                    interactionSpec, individuals);
            cage = new Cage(rankingIdentifier, interactionSpec, individuals,
                    matrix);
            // Initialize the ranking properly:
            cage.ranks.permute(DefaultAutoRanker.rank(cage), null);
            Date latestDate = db.getLatestRankingDateSaved(rankingIdentifier);
            String initializeUITitle = "Initializing ranking";
            if (latestDate == null) {
                String text = "<html>" + "Babase has no exiting ranking for "
                        + rankingIdentifier + " or before.<br>"
                        + "Ranking will be initialized using default rules."
                        + "</html>";
                String[] options = { "Proceed", "Cancel" };
                int n = JOptionPane
                        .showOptionDialog(this, text, initializeUITitle,
                                JOptionPane.YES_NO_OPTION,
                                JOptionPane.QUESTION_MESSAGE, null, options,
                                options[0]);
                return (n == 0) ? cage : null;
            } else if (latestDate.equals(cage.rankingId.startDate)
                    && Babase.isEndOfMonth(cage.rankingId.endDate)
                    && Babase.isInSameMonth(cage.rankingId.startDate,
                            cage.rankingId.endDate)) {
                String text = "<html>"
                        + "Babase has an existing ranking for "
                        + rankingIdentifier
                        + ".<br>"
                        + "Load the existing ranking, or initalize using default rules?"
                        + "</html>";
                String[] options = { "Load", "Default", "Cancel" };
                int n = JOptionPane
                        .showOptionDialog(this, text, initializeUITitle,
                                JOptionPane.YES_NO_CANCEL_OPTION,
                                JOptionPane.QUESTION_MESSAGE, null, options,
                                options[0]);
                if (n == 0) {
                    RankingInfo existingRankingInfo = db
                            .getRankingInfo(cage.rankingId);
                    IncorporatorUI.incorporate(this,
                            "Inconsistency with existing ranking", cage,
                            existingRankingInfo, true);
                    return cage;
                } else {
                    return (n == 1) ? cage : null;
                }
            } else {
                String text = "<html>"
                        + "Babase has no existing ranking for "
                        + rankingIdentifier
                        + ";<br>"
                        + "the closest preceding ranking is for "
                        + Babase.dateToString(latestDate)
                        + ".<br>"
                        + "After initalizing ranking using default rules,<br>"
                        + "would you like to incorporate the preceding ranking?"
                        + "</html>";
                String[] options = { "Yes", "No", "Cancel" };
                int n = JOptionPane
                        .showOptionDialog(this, text, initializeUITitle,
                                JOptionPane.YES_NO_CANCEL_OPTION,
                                JOptionPane.QUESTION_MESSAGE, null, options,
                                options[0]);
                if (n == 0) {
                    RankingInfo existingRankingInfo = db
                            .getRankingInfo(new RankingIdentifier(
                                    cage.rankingId.group,
                                    cage.rankingId.rankType, latestDate));
                    IncorporatorUI.incorporate(this,
                            "Incorporate previous ranking", cage,
                            existingRankingInfo, false);
                    return cage;
                } else {
                    return (n == 1) ? cage : null;
                }
            }
        } catch (SQLException e) {
            JOptionPane.showMessageDialog(this, e.getMessage());
            return null;
        }
    }
}
