package babase.ranker;

import info.clearthought.layout.TableLayout;

import java.awt.Container;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.sql.SQLException;
import java.util.Date;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import babase.db.Babase;
import babase.db.Group;
import babase.db.RankType;
import babase.db.RankingIdentifier;
import babase.db.RankingInfo;

import com.toedter.calendar.JDateChooser;

public class LoadUI {

    private final MainUI _mainUI;

    private final Babase _db;

    private final Cage _cage;

    private int[] _newRanks = null;

    public LoadUI(MainUI mainUI, Babase db, Cage cage) {
        _mainUI = mainUI;
        _db = db;
        _cage = cage;
        return;
    }

    public int[] getNewRanks() {
        return _newRanks;
    }

    public void start(final ActionListener loadedActionListener) {
        final JDialog toLoadDialog = new JDialog((Frame) _mainUI,
                "Load/incorporate ranking from Babase", true);
        Date latestDate = null;
        RankType[] rankTypes = null;
        Group[] groups = null;
        try {
            latestDate = _db.getLatestRankingDateSaved(_cage.rankingId);
            rankTypes = _db.getRankTypes();
            groups = _db.getGroups();
        } catch (SQLException e) {
            Utils.showErrorDialog(_mainUI, "Babase connection problem: "
                    + e.getMessage());
            return;
        }
        String info = null;
        if (latestDate == null) {
            info = "No existing ranking for " + _cage.rankingId + " or before.";
        } else if (latestDate.equals(_cage.rankingId.startDate)) {
            info = "Existing ranking found for " + _cage.rankingId + ".";
        } else {
            info = "No existing ranking for " + _cage.rankingId + ";<br>"
                    + "closest preceding one found on"
                    + Babase.dateToString(latestDate) + ".";
        }
        JLabel label = new JLabel("<html>" + info
                + "<br>Specify the ranking to load:</html>");
        final JComboBox rankTypeBox = new JComboBox(rankTypes);
        rankTypeBox.setSelectedIndex(RankType.findRankType(rankTypes,
                _cage.rankingId.rankType));
        final JComboBox groupBox = new JComboBox(groups);
        groupBox.setSelectedIndex(Group
                .findGroup(groups, _cage.rankingId.group));
        final JDateChooser rankStartDateChooser = new JDateChooser(
                "yyyy-MM-dd", "####-##-##", '_');
        rankStartDateChooser
                .setDate((latestDate == null) ? _cage.rankingId.startDate
                        : latestDate);
        JButton cancelButton = new JButton("Cancel");
        cancelButton.setMnemonic(KeyEvent.VK_C);
        cancelButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                toLoadDialog.dispose();
                return;
            }
        });
        JButton loadButton = new JButton("Load");
        loadButton.setMnemonic(KeyEvent.VK_L);
        loadButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                RankingIdentifier rid = new RankingIdentifier((Group) groupBox
                        .getSelectedItem(), (RankType) rankTypeBox
                        .getSelectedItem(), rankStartDateChooser.getDate());
                toLoadDialog.dispose();
                RankingInfo rankingInfo = null;
                try {
                    rankingInfo = _db.getRankingInfo(rid);
                } catch (SQLException sqlException) {
                    Utils.showErrorDialog(_mainUI,
                            "Babase connection problem: "
                                    + sqlException.getMessage());
                    return;
                }
                if (rankingInfo == null) {
                    Utils.showErrorDialog(_mainUI, "No existing ranking for "
                            + rid + ".");
                    return;
                }
                IncorporatorUI.incorporate(_mainUI, "Confirm changes to your ranking",
                        _cage, rankingInfo, false);
                return;
            }
        });
        double border = 10;
        double[] tableLayoutWidths = { border, TableLayout.PREFERRED, border,
                TableLayout.PREFERRED, border };
        double[] tableLayoutHeights = { border, TableLayout.PREFERRED, border,
                TableLayout.PREFERRED, border, TableLayout.PREFERRED, border,
                TableLayout.PREFERRED, border, TableLayout.PREFERRED, border };
        Container pane = toLoadDialog.getContentPane();
        pane.setLayout(new TableLayout(tableLayoutWidths, tableLayoutHeights));
        pane.add(label, "1, 1, 3, 1, c, t");
        pane.add(new JLabel("Rank type: "), "1, 3, r, c");
        pane.add(rankTypeBox, "3, 3, f, c");
        pane.add(new JLabel("Group: "), "1, 5, r, c");
        pane.add(groupBox, "3, 5, f, c");
        pane.add(new JLabel("Ranking start date: "), "1, 7, r, c");
        pane.add(rankStartDateChooser, "3, 7, f, c");
        pane.add(cancelButton, "1, 9, c, c");
        pane.add(loadButton, "3, 9, c, c");
        toLoadDialog.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        toLoadDialog.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                toLoadDialog.dispose();
                return;
            }
        });
        toLoadDialog.pack();
        toLoadDialog.setLocationRelativeTo(null);
        toLoadDialog.setVisible(true);
        return;
    }

}
