package babase.ranker.auto;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.EnumMap;

import babase.db.Individual;
import babase.db.Individual.IndividualClassification;
import babase.ranker.Cage;

/**
 * The <code>DefaultAutoRanker</code> implements the following rule for
 * ordering individuals. See {@link IndividualClassification} for precise
 * definition of enumeration constants below; the classification is computed
 * with respect to the ranking start date.
 * 
 * First come the adult males (<code>ADULT_MALE</code>) in reverse age order
 * (youngest to oldest).
 * 
 * Then come subadult males (<code>SUBADULT_MALE</code>) in age order
 * (oldest to youngest).
 * 
 * Then come adult females (<code>ADULT_FEMALE</code>) in alphabetical order
 * of <code>sname</code>s.
 * 
 * Then come juvenile males (<code>JUV_MALE</code>) in age order (oldest to
 * youngest).
 * 
 * Then come juvenile females (<code>JUV_FEMALE</code>) in age order (oldest
 * to youngest).
 * 
 * Finally, any one who do not fall into any of the above classifications (<code>UNKNOWN</code>).
 * 
 * Whenever there is a tie, we go by alpabetical order of <code>sname</code>s.
 * 
 * @author junyang
 * 
 */
public class DefaultAutoRanker {

    private static final EnumMap<IndividualClassification, Integer> INDIVIDUAL_CLASSIFICATION_ORDER;
    static {
        INDIVIDUAL_CLASSIFICATION_ORDER = new EnumMap<IndividualClassification, Integer>(
                IndividualClassification.class);
        INDIVIDUAL_CLASSIFICATION_ORDER.put(
                IndividualClassification.ADULT_MALE, 0);
        INDIVIDUAL_CLASSIFICATION_ORDER.put(
                IndividualClassification.SUBADULT_MALE, 1);
        INDIVIDUAL_CLASSIFICATION_ORDER.put(
                IndividualClassification.ADULT_FEMALE, 2);
        INDIVIDUAL_CLASSIFICATION_ORDER.put(IndividualClassification.JUV_MALE,
                3);
        INDIVIDUAL_CLASSIFICATION_ORDER.put(
                IndividualClassification.JUV_FEMALE, 4);
        INDIVIDUAL_CLASSIFICATION_ORDER
                .put(IndividualClassification.UNKNOWN, 5);
    }

    public static int[] rank(final Cage cage) {
        Integer[] newRank = new Integer[cage.ranks.size()];
        for (int i = 0; i < cage.ranks.size(); i++) {
            newRank[i] = cage.ranks.get(i);
        }
        Arrays.sort(newRank, new Comparator<Integer>() {
            public int compare(Integer i, Integer j) {
                Individual a = cage.individuals[i];
                Individual b = cage.individuals[j];
                if (a == b)
                    return 0;
                Date rankDate = cage.rankingId.startDate;
                IndividualClassification ac = a.getClassification(rankDate);
                IndividualClassification bc = b.getClassification(rankDate);
                if (ac == bc) {
                    if (ac == IndividualClassification.ADULT_MALE) {
                        // Youngest ranked first:
                        if (a.birth.before(b.birth)) {
                            return 1;
                        } else if (a.birth.after(b.birth)) {
                            return -1;
                        } else {
                            return a.sname.compareTo(b.sname);
                        }
                    } else if (ac == IndividualClassification.ADULT_FEMALE) {
                        // Alphabetical order:
                        return a.sname.compareTo(b.sname);
                    } else {
                        // Oldest ranked first:
                        if (a.birth.before(b.birth)) {
                            return -1;
                        } else if (a.birth.after(b.birth)) {
                            return 1;
                        } else {
                            return a.sname.compareTo(b.sname);
                        }
                    }
                } else {
                    return INDIVIDUAL_CLASSIFICATION_ORDER.get(ac).compareTo(
                            INDIVIDUAL_CLASSIFICATION_ORDER.get(bc));
                }
            }
        });
        int[] newIntRank = new int[newRank.length];
        for (int i = 0; i < newRank.length; i++)
            newIntRank[i] = newRank[i].intValue();
        return newIntRank;
    }
}