package babase.db;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Properties;
import java.util.Vector;

public class Babase {

    public static class LoadFromXMLException extends Exception {

        private static final long serialVersionUID = 1L;

        public LoadFromXMLException(String message) {
            super(message);
        }

        public LoadFromXMLException(String message, Throwable cause) {
            super(message, cause);
        }

    }

    public static class DumpToXMLException extends Exception {

        private static final long serialVersionUID = 1L;

        public DumpToXMLException(String message) {
            super(message);
        }

        public DumpToXMLException(String message, Throwable cause) {
            super(message, cause);
        }

    }

    public static String dateToString(Date d) {
        return (new SimpleDateFormat("yyyy-MM-dd")).format(d);
    }

    public static String datetimeToString(Date d) {
        return (new SimpleDateFormat("yyyy-MM-dd HH:mm:ss EEE")).format(d);
    }

    public static String dateToXMLString(Date d) {
        return (new SimpleDateFormat("yyyy-MM-dd")).format(d);
    }

    public static boolean isStartOfMonth(Date d) {
        Calendar c = Calendar.getInstance();
        c.setTime(d);
        return c.get(Calendar.DAY_OF_MONTH) == 1;
    }

    public static boolean isEndOfMonth(Date d) {
        Calendar c = Calendar.getInstance();
        c.setTime(d);
        int currentMonth = c.get(Calendar.MONTH);
        c.add(Calendar.DAY_OF_MONTH, 1);
        return currentMonth != c.get(Calendar.MONTH);
    }

    public static boolean isInSameMonth(Date d1, Date d2) {
        Calendar c = Calendar.getInstance();
        c.setTime(d1);
        int year1 = c.get(Calendar.YEAR);
        int month1 = c.get(Calendar.MONTH);
        c.setTime(d2);
        int year2 = c.get(Calendar.YEAR);
        int month2 = c.get(Calendar.MONTH);
        return year1 == year2 && month1 == month2;
    }

    public static Date endOfMonth(Date d) {
        Calendar c = Calendar.getInstance();
        c.setTime(d);
        c.add(Calendar.MONTH, 1);
        c.set(Calendar.DAY_OF_MONTH, 1);
        c.add(Calendar.DAY_OF_MONTH, -1);
        return c.getTime();
    }

    public static Date xmlStringToDate(String s) throws LoadFromXMLException {
        try {
            return (new SimpleDateFormat("yyyy-MM-dd")).parse(s);
        } catch (ParseException e) {
            throw new LoadFromXMLException("'" + s
                    + "' is not formatted correctly as a date", e);
        }
    }

    private static String dateToSQLLiteral(Date d) {
        return "'" + (new SimpleDateFormat("yyyy-MM-dd")).format(d) + "'";
    }

    private Connection _connection = null;

    private ArrayList<Group> _cachedGroups = null;

    private ArrayList<String> _cachedClasses = null;

    private ArrayList<RankType> _cachedRankTypes = null;

    public Babase() throws ClassNotFoundException {
        Class.forName("org.postgresql.Driver");
    }

    public void connect(String host, String port, String user, String pass,
            String db, Boolean ssl) throws SQLException {
        if (_connection != null)
            disconnect();
        Properties props = new Properties();
        props.setProperty("user", user);
        props.setProperty("password", pass);
        if (ssl) {
            props.setProperty("ssl", "true");
            props.setProperty("sslfactory",
                    "org.postgresql.ssl.NonValidatingFactory");
        }
        _connection = DriverManager.getConnection("jdbc:postgresql://" + host
                + ":" + port + "/" + db, props);
    }

    public boolean isConnected() {
        return (_connection != null);
    }

    public void disconnect() throws SQLException {
        _connection.close();
        _connection = null;
        _cachedGroups = null;
        _cachedClasses = null;
        _cachedRankTypes = null;
    }

    public RankType[] getRankTypes() throws SQLException {
        if (_cachedRankTypes == null) {
            ArrayList<RankType> rankTypes = new ArrayList<RankType>();
            Statement s = _connection.createStatement();
            ResultSet rs = s
                    .executeQuery("SELECT DISTINCT rnktype, descr FROM rnktypes ORDER BY rnktype");
            while (rs.next()) {
                rankTypes.add(new RankType(rs.getString(1), rs.getString(2)));
            }
            rs.close();
            s.close();
            _cachedRankTypes = rankTypes;
        }
        return _cachedRankTypes.toArray(new RankType[_cachedRankTypes.size()]);
    }

    public Group[] getGroups() throws SQLException {
        if (_cachedGroups == null) {
            ArrayList<Group> groups = new ArrayList<Group>();
            Statement s = _connection.createStatement();
            ResultSet rs = s
                    .executeQuery("SELECT DISTINCT gid, name FROM groups ORDER BY gid;");
            while (rs.next()) {
                groups.add(new Group(rs.getBigDecimal(1), rs.getString(2)));
            }
            rs.close();
            s.close();
            _cachedGroups = groups;
        }
        return _cachedGroups.toArray(new Group[_cachedGroups.size()]);
    }

    public String[] getInteractionClasses() throws SQLException {
        if (_cachedClasses == null) {
            ArrayList<String> classes = new ArrayList<String>();
            Statement s = _connection.createStatement();
            ResultSet rs = s
                    .executeQuery("SELECT DISTINCT class FROM acts ORDER BY class;");
            while (rs.next()) {
                classes.add(rs.getString(1));
            }
            rs.close();
            s.close();
            _cachedClasses = classes;
        }
        return _cachedClasses.toArray(new String[_cachedClasses.size()]);
    }

    public String getSnamesToRankQuery(RankingIdentifier rankingId)
            throws SQLException {
        Statement s = _connection.createStatement();
        ResultSet rs = s
                .executeQuery("SELECT query FROM rnktypes WHERE rnktype = "
                        + rankingId.rankType.toSQLLiteral());
        rs.next();
        String query = rs.getString(1);
        rs.close();
        s.close();
        query = query.replace("%s", dateToSQLLiteral(rankingId.startDate));
        query = query.replace("%f", dateToSQLLiteral(rankingId.endDate));
        query = query.replace("%g", rankingId.group.toSQLLiteral());
        return query;
    }

    public Individual[] getIndividuals(RankingIdentifier rankingId)
            throws SQLException {
        String snamesToRankQuery = getSnamesToRankQuery(rankingId);
        Statement s = _connection.createStatement();
        ResultSet rs = s
                .executeQuery("SELECT bioid, sname, name, pid, birth, bstatus, sex,"
                        + " matgrp, statdate, status, dcause, dcausestatus,"
                        + " (SELECT MIN(matured) FROM maturedates WHERE sname = biograph.sname),"
                        + " (SELECT MIN(ranked) FROM rankdates WHERE sname = biograph.sname)"
                        + " FROM biograph WHERE sname IN (" + snamesToRankQuery
                        + ") ORDER BY sname");
        ArrayList<Individual> individuals = new ArrayList<Individual>();
        while (rs.next()) {
            individuals.add(new Individual(rs.getString(2),
                    (rs.getObject(1) == null) ? null : new Integer(rs.getInt(1)),
                    rs.getString(3),
                    rs.getString(4),
                    rs.getDate(5),
                    (rs.getObject(6) == null) ? null : new Integer(rs.getInt(6)),
                    (rs.getString(7) == null) ? null : rs.getString(7)
                            .toCharArray()[0],
                    rs.getBigDecimal(8),
                    rs.getDate(9),
                    (rs.getObject(10) == null) ? null : new Integer(rs.getInt(10)),
                    (rs.getObject(11) == null) ? null : new Integer(rs.getInt(11)),
                    (rs.getObject(12) == null) ? null : new Integer(rs.getInt(12)),
                    rs.getDate(13),
                    rs.getDate(14)));
        }
        rs.close();
        s.close();
        return individuals.toArray(new Individual[individuals.size()]);
    }

    public int[][] getInteractionMatrix(RankingIdentifier rankingId,
            InteractionSpec interactionSpec, Individual[] individuals)
            throws SQLException {
        String query = getSnamesToRankQuery(rankingId);
        int[][] matrix = new int[individuals.length][individuals.length];
        for (int i = 0; i < individuals.length; i++)
            for (int j = 0; j < individuals.length; j++)
                matrix[i][j] = 0;
        Statement s = _connection.createStatement();
        ResultSet rs = s
                .executeQuery("SELECT actor, actee, COUNT(*) FROM actor_actees"
                        + " WHERE actor IN (" + query + ")" + " AND actee IN ("
                        + query + ")" + " AND date >= "
                        + dateToSQLLiteral(interactionSpec.startDate)
                        + " AND date <= "
                        + dateToSQLLiteral(interactionSpec.endDate)
                        + " AND act IN (SELECT act FROM acts WHERE class = '"
                        + interactionSpec.interactionClass + "') "
                        + " GROUP BY actor, actee");
        while (rs.next()) {
            String actor = rs.getString(1);
            String actee = rs.getString(2);
            int count = rs.getInt(3);
            int row = Individual.findIndividualBySname(actor, individuals);
            int col = Individual.findIndividualBySname(actee, individuals);
            assert (row != -1 && col != -1);
            matrix[row][col] = count;
        }
        return matrix;
    }

    public void writeRankingInfo(RankingIdentifier rid,
            Individual[] individuals, int[] ranks) throws SQLException {
        // Below is a restriction for now; may be lifted in the future:
        assert (Babase.isStartOfMonth(rid.startDate)
                && Babase.isEndOfMonth(rid.endDate) && Babase.isInSameMonth(
                rid.startDate, rid.endDate));
        boolean autoCommit = _connection.getAutoCommit();
        _connection.setAutoCommit(false);
        Statement s = _connection.createStatement();
        s.executeUpdate("DELETE FROM ranks" + " WHERE rnkdate = "
                + dateToSQLLiteral(rid.startDate) + " AND grp = "
                + rid.group.toSQLLiteral() + " AND rnktype = "
                + rid.rankType.toSQLLiteral());
        for (int i = 0; i < ranks.length; i++) {
            String sname = individuals[ranks[i]].sname;
            s
                    .executeUpdate("INSERT INTO ranks(rnkdate, grp, rnktype, sname, rank)"
                            + " VALUES ("
                            + dateToSQLLiteral(rid.startDate)
                            + ", "
                            + rid.group.toSQLLiteral()
                            + ", "
                            + rid.rankType.toSQLLiteral()
                            + ", '"
                            + sname
                            + "', " + (i + 1) + ")");
        }
        _connection.commit();
        _connection.setAutoCommit(autoCommit);
        return;
    }

    public RankingInfo getRankingInfo(RankingIdentifier rid)
            throws SQLException {
        if (!isStartOfMonth(rid.startDate) || !isEndOfMonth(rid.endDate)
                || !isInSameMonth(rid.startDate, rid.endDate)) {
            return null;
        }
        Statement s = _connection.createStatement();
        ResultSet rs = s
                .executeQuery("SELECT sname, rank FROM ranks WHERE grp = "
                        + rid.group.toSQLLiteral() + " AND rnktype = "
                        + rid.rankType.toSQLLiteral() + " AND rnkdate = "
                        + dateToSQLLiteral(rid.startDate));
        Vector<String> snames = new Vector<String>();
        Vector<Integer> ranks = new Vector<Integer>();
        while (rs.next()) {
            snames.add(rs.getString(1));
            ranks.add(rs.getInt(2) - 1);
        }
        return (snames.isEmpty()) ? null : new RankingInfo(rid, snames, ranks);
    }

    public Date getLatestRankingDateSaved(RankingIdentifier rid)
            throws SQLException {
        Statement s = _connection.createStatement();
        ResultSet rs = s
                .executeQuery("SELECT MAX(rnkdate) FROM ranks WHERE grp = "
                        + rid.group.toSQLLiteral() + " AND rnktype = "
                        + rid.rankType.toSQLLiteral() + " AND rnkdate <= "
                        + dateToSQLLiteral(rid.startDate));
        Date latestDate = null;
        while (rs.next()) {
            latestDate = rs.getDate(1);
        }
        rs.close();
        s.close();
        return latestDate;
    }

}
