package babase.ranker.ranking;

/**
 * An event representing a change made to a {@link Ranking} object.
 * 
 * @author junyang
 * 
 */
public abstract class RankingChangeEvent {

    /**
     * The object that caused this <code>RankingChangedEvent</code>.
     */
    public final Object source;

    /**
     * The {@link Ranking} object that has been changed.
     */
    public final Ranking ranks;

    /**
     * @param source
     * @param ranks
     */
    public RankingChangeEvent(Object source, Ranking ranks) {
        this.source = source;
        this.ranks = ranks;
        return;
    }

    /**
     * An event representing the operation of swapping two individuals' ranks,
     * caused by {@link Ranking#swap(int, int, Object)}.
     * 
     * @author junyang
     * 
     */
    public static class Swap extends RankingChangeEvent {

        /**
         * One of the ranks (0-based) involved in the swap.
         */
        public final int i;

        /**
         * The other rank (0-based) involved in the swap.
         */
        public final int j;

        /**
         * @param swapper
         * @param ranks
         * @param i
         * @param j
         */
        public Swap(Object swapper, Ranking ranks, int i, int j) {
            super(swapper, ranks);
            this.i = i;
            this.j = j;
            return;
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Object#toString()
         */
        public String toString() {
            // Output uses 1-based index:
            return "SWAP(" + (i + 1) + ", " + (j + 1) + ")";
        }

    }

    /**
     * An event representing the operation of "sliding" an individual from one
     * rank to another, caused by {@link Ranking#slide(int, int, Object)}.
     * 
     * @author junyang
     * 
     */
    public static class Slide extends RankingChangeEvent {

        /**
         * The original rank (0-based) of the individual being slided.
         */
        public final int from;

        /**
         * The destination rank (0-based) to the individual being slided.
         */
        public final int to;

        /**
         * @param slider
         * @param ranks
         * @param from
         * @param to
         */
        public Slide(Object slider, Ranking ranks, int from, int to) {
            super(slider, ranks);
            this.from = from;
            this.to = to;
            return;
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Object#toString()
         */
        public String toString() {
            // Output uses 1-based index:
            return "SLIDE(" + (from + 1) + "->" + (to + 1) + ")";
        }

    }

    /**
     * An event representing the operation of permuting the entire ranking,
     * caused by {@link Ranking#permute(int[], Object)}.
     * 
     * @author junyang
     * 
     */
    public static class Permute extends RankingChangeEvent {

        /**
         * The state of the ranking immediately before the permutation.
         */
        public final int[] before;

        /**
         * The state of the ranking immediately after the permutation.
         */
        public final int[] after;

        /**
         * @param permuter
         * @param ranks
         * @param before
         * @param after
         */
        public Permute(Object permuter, Ranking ranks, int[] before, int[] after) {
            super(permuter, ranks);
            this.before = before;
            this.after = after;
            return;
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Object#toString()
         */
        public String toString() {
            String s = "PERMUTE([";
            for (int i = 0; i < before.length; i++) {
                if (i > 0)
                    s += ", ";
                // Output uses 1-based index:
                s += (before[i] + 1);
            }
            s += "] -> [";
            for (int i = 0; i < after.length; i++) {
                if (i > 0)
                    s += ", ";
                // Output uses 1-based index:
                s += (after[i] + 1);
            }
            s += ']';
            return s;
        }

    }

}
