package babase.db;

import java.math.BigDecimal;
import java.util.Date;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import babase.db.Babase.LoadFromXMLException;

public class Individual {

    /**
     * With respect to the particular date <code>d</code>, we classify
     * individuals as follows:
     * 
     * <code>ADULT_FEMALE</code>: <code>sex == 'F'</code> and
     * <code>matured <= d</code>.
     * 
     * <code>JUV_FEMALE</code>: <code>sex == 'F'</code>, but not
     * <code>ADULT_FEMALE</code>.
     * 
     * <code>ADULT_MALE</code>: <code>sex == 'M'</code> and
     * <code>rankd</code> (rank attainment date) <code><= d</code>.
     * 
     * <code>SUBADULT_MALE</code>: <code>sex == 'M'</code> and
     * <code>matured <= d</code>, but not <code>ADULT_MALE</code>.
     * 
     * <code>JUV_MALE</code>: <code>sex == 'M'</code>, but not
     * <code>ADULT_MALE</code> or <code>SUBADULT_MALE</code>.
     * 
     * <code>UNKNOWN</code>: none of the above.
     * 
     * @author junyang
     * 
     */
    public enum IndividualClassification {
        ADULT_FEMALE, JUV_FEMALE, ADULT_MALE, SUBADULT_MALE, JUV_MALE, UNKNOWN
    };

    public static int findIndividualBySname(String sname,
            Individual[] individuals) {
        for (int i = 0; i < individuals.length; i++) {
            if (sname.equals(individuals[i].sname))
                return i;
        }
        return -1;
    }

    public final String sname;

    public final Integer bioId;

    public final String name;

    public final String pid;

    public final Date birth;

    public final Integer bStatus;

    public final Character sex;

    public final BigDecimal matGrp;

    public final Date statDate;

    public final Integer status;

    public final Integer dCause;

    public final Integer dCauseConfidence;

    public final Date matured;

    public final Date ranked;

    public Individual(String sname, Integer bioId, String name, String pid,
            Date birth, Integer bStatus, Character sex, BigDecimal matGrp,
            Date statDate, Integer status, Integer dCause, Integer dCauseConfidence,
            Date matured, Date ranked) {
        assert (sname != null && bioId != null && name != null && birth != null);
        this.sname = sname;
        this.bioId = bioId;
        this.name = name;
        this.pid = pid;
        this.birth = birth;
        this.bStatus = bStatus;
        this.sex = sex;
        this.matGrp = matGrp;
        this.statDate = statDate;
        this.status = status;
        this.dCause = dCause;
        this.dCauseConfidence = dCauseConfidence;
        this.matured = matured;
        this.ranked = ranked;
    }

    public Individual(Individual individual) {
        this.sname = individual.sname;
        this.bioId = individual.bioId;
        this.name = individual.name;
        this.pid = individual.pid;
        this.birth = individual.birth;
        this.bStatus = individual.bStatus;
        this.sex = individual.sex;
        this.matGrp = individual.matGrp;
        this.statDate = individual.statDate;
        this.status = individual.status;
        this.dCause = individual.dCause;
        this.dCauseConfidence = individual.dCauseConfidence;
        this.matured = individual.matured;
        this.ranked = individual.ranked;
    }

    /**
     * Return the classification of the individual as of a particular date.
     * 
     * @param rankDate
     *            The date on which the classification is based.
     * @return The classification.
     */
    public IndividualClassification getClassification(final Date rankDate) {
        if (sex != null && sex.charValue() == 'F') {
            if (matured != null && !rankDate.before(matured)) {
                return IndividualClassification.ADULT_FEMALE;
            } else {
                return IndividualClassification.JUV_FEMALE;
            }
        } else if (sex != null && sex.charValue() == 'M') {
            if (ranked != null && !rankDate.before(ranked)) {
                return IndividualClassification.ADULT_MALE;
            } else if (matured != null && !rankDate.before(matured)) {
                return IndividualClassification.SUBADULT_MALE;
            } else {
                return IndividualClassification.JUV_MALE;
            }
        }
        return IndividualClassification.UNKNOWN;
    }

    public Element toXML(Document d) {
        Element e = d.createElement("individual");
        e.setAttribute("sname", sname);
        e.setAttribute("bioId", bioId.toString());
        e.setAttribute("name", name);
        if (pid != null)
            e.setAttribute("pid", pid);
        e.setAttribute("birth", Babase.dateToXMLString(birth));
        if (bStatus != null)
            e.setAttribute("bStatus", bStatus.toString());
        if (sex != null)
            e.setAttribute("sex", sex.toString());
        if (matGrp != null)
            e.setAttribute("matGrp", matGrp.toString());
        if (statDate != null)
            e.setAttribute("statDate", Babase.dateToXMLString(statDate));
        if (status != null)
            e.setAttribute("status", status.toString());
        if (dCause != null)
            e.setAttribute("dCause", dCause.toString());
        if (dCauseConfidence != null)
            e.setAttribute("dCauseConfidence", dCauseConfidence.toString());
        if (matured != null)
            e.setAttribute("matured", Babase.dateToXMLString(matured));
        if (ranked != null)
            e.setAttribute("ranked", Babase.dateToXMLString(ranked));
        return e;
    }

    public static Individual fromXML(Element e) throws LoadFromXMLException {
        assert (e.getNodeName().equals("individual"));
        return new Individual(e.getAttribute("sname"), new Integer(e
                .getAttribute("bioId")), e.getAttribute("name"), e
                .hasAttribute("pid") ? e.getAttribute("pid") : null, Babase
                .xmlStringToDate(e.getAttribute("birth")), e
                .hasAttribute("bStatus") ? new Integer(e
                .getAttribute("bStatus")) : null,
                e.hasAttribute("sex") ? new Character(e.getAttribute("sex")
                        .charAt(0)) : null,
                e.hasAttribute("matGrp") ? new BigDecimal(e
                        .getAttribute("matGrp")) : null, e
                        .hasAttribute("statDate") ? Babase.xmlStringToDate(e
                        .getAttribute("statDate")) : null, e
                        .hasAttribute("status") ? new Integer(e
                        .getAttribute("status")) : null, e
                        .hasAttribute("dCause") ? new Integer(e
                        .getAttribute("dCause")) : null, e
                        .hasAttribute("dCauseConfidence") ? new Integer(e
                        .getAttribute("dCauseConfidence")) : null, e
                        .hasAttribute("matured") ? Babase.xmlStringToDate(e
                        .getAttribute("matured")) : null, e
                        .hasAttribute("ranked") ? Babase.xmlStringToDate(e
                        .getAttribute("ranked")) : null);
    }
}
