package babase.db;

import java.util.Date;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import babase.db.Babase.LoadFromXMLException;

/**
 * <code>RankingIdentifier</code> uniquely identifies a "ranking" in Babase.
 * This identifier can be used to retrieve or store a ranking in Babase. As an
 * indentifer, instances of this class do not carry actual ranking contents
 * (i.e., which individual is ranked at what position).
 * 
 * @author junyang
 * 
 */
public class RankingIdentifier {
    /**
     * The group that this ranking is for.
     */
    public final Group group;

    /**
     * The type of the ranking.
     */
    public final RankType rankType;

    /**
     * The ranking start date (inclusive).
     */
    public final Date startDate;

    /**
     * The ranking end date (inclusive).
     */
    public final Date endDate;

    /**
     * Constructor.
     * 
     * @param group
     * @param rankType
     * @param startDate
     *            Since <code>Date</code> is mutable, this constructor will a
     *            copy of the input object.
     * @param endDate
     *            Since <code>Date</code> is mutable, this constructor will a
     *            copy of the input object.
     */
    public RankingIdentifier(Group group, RankType rankType, Date startDate,
            Date endDate) {
        this.group = group;
        this.rankType = rankType;
        this.startDate = (Date) startDate.clone();
        this.endDate = (Date) endDate.clone();
        return;
    }

    /**
     * A convenience constructor. <code>endDate</code> will be automatically
     * set to the last day of the month of <code>startDate</code>.
     * 
     * @param group
     * @param rankType
     * @param startDate
     *            Since <code>Date</code> is mutable, this constructor will a
     *            copy of the input object.
     */
    public RankingIdentifier(Group group, RankType rankType, Date startDate) {
        this.group = group;
        this.rankType = rankType;
        this.startDate = (Date) startDate.clone();
        this.endDate = Babase.endOfMonth(startDate);
        return;
    }

    public String toString() {
        return group.name + " " + rankType.type + " "
                + Babase.dateToString(startDate) + "~"
                + Babase.dateToString(endDate);
    }

    public Element toXML(Document d) {
        Element e = d.createElement("rankingId");
        e.setAttribute("startDate", Babase.dateToXMLString(startDate));
        e.setAttribute("endDate", Babase.dateToXMLString(endDate));
        e.appendChild(rankType.toXML(d));
        e.appendChild(group.toXML(d));
        return e;
    }

    public static RankingIdentifier fromXML(Element e)
            throws LoadFromXMLException {
        assert (e.getNodeName().equals("rankingId"));
        Date startDate = Babase.xmlStringToDate(e.getAttribute("startDate"));
        Date endDate = Babase.xmlStringToDate(e.getAttribute("endDate"));
        RankType rankType = null;
        Group group = null;
        for (Node child = e.getFirstChild(); child != null; child = child
                .getNextSibling()) {
            if (child.getNodeType() != Node.ELEMENT_NODE)
                continue;
            Element ce = (Element) child;
            if (ce.getNodeName().equals("rankType")) {
                rankType = RankType.fromXML(ce);
            } else if (ce.getNodeName().equals("group")) {
                group = Group.fromXML(ce);
            } else {
                assert (false);
            }
        }
        return new RankingIdentifier(group, rankType, startDate, endDate);
    }

}
