package babase.ranker;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Calendar;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

import babase.db.Babase;
import babase.db.Individual;
import babase.db.InteractionSpec;
import babase.db.RankingIdentifier;
import babase.db.Babase.DumpToXMLException;
import babase.db.Babase.LoadFromXMLException;
import babase.ranker.ranking.Ranking;

public class Cage {

    private static DocumentBuilderFactory _factory = DocumentBuilderFactory
            .newInstance();

    public final RankingIdentifier rankingId;

    public final InteractionSpec interactionSpec;

    public final IndividualHeader[] individuals;

    public final Ranking ranks;

    public final int[][] matrix;

    public Cage(RankingIdentifier rankingId, InteractionSpec interactionSpec,
            Individual[] individuals, int[] ranks, int[][] matrix) {
        this.rankingId = rankingId;
        this.interactionSpec = interactionSpec;
        this.individuals = new IndividualHeader[individuals.length];
        for (int i = 0; i < individuals.length; i++) {
            this.individuals[i] = new IndividualHeader(individuals[i]);
        }
        if (ranks == null) {
            this.ranks = new Ranking(individuals.length);
        } else {
            this.ranks = new Ranking(ranks);
        }
        this.matrix = matrix.clone();
        for (int i = 0; i < matrix.length; i++)
            this.matrix[i] = matrix[i].clone();
        return;
    }

    public Cage(RankingIdentifier rankingIdentifier,
            InteractionSpec interactionSpec, Individual[] individuals,
            int[][] matrix) {
        this(rankingIdentifier, interactionSpec, individuals, null, matrix);
        return;
    }

    private void printRepeat(PrintStream out, int times, String s) {
        for (int i = 0; i < times; i++)
            out.print(s);
        return;
    }

    public void print(PrintStream out) {
        out.println(rankingId);
        out.println(interactionSpec);
        out.println("Printed on "
                + Babase.datetimeToString(Calendar.getInstance().getTime()));
        out.println();
        int maxHeaderLength = 0;
        for (int i = 0; i < individuals.length; i++) {
            int len = individuals[i].sname.length();
            if (maxHeaderLength < len)
                maxHeaderLength = len;
        }
        int maxRankWidth = Integer.toString(ranks.size() + 1).length();
        int maxEntryWidth = maxRankWidth;
        for (int i = 0; i < matrix.length; i++) {
            for (int j = 0; j < matrix[i].length; j++) {
                int len = Integer.toString(matrix[i][j]).length();
                if (maxEntryWidth < len)
                    maxEntryWidth = len;
            }
        }
        maxEntryWidth++;
        for (int i = 0; i < maxHeaderLength; i++) {
            printRepeat(out, maxHeaderLength + 1 + maxRankWidth + 1, " ");
            for (int j = 0; j < individuals.length; j++) {
                final String sname = individuals[ranks.get(j)].sname;
                if (sname.length() > i)
                    out.format("%1$" + maxEntryWidth + "s", ""
                            + sname.charAt(i));
                else
                    printRepeat(out, maxEntryWidth, " ");
            }
            out.println();
        }
        printRepeat(out, maxHeaderLength + 1 + maxRankWidth + 1, " ");
        for (int i = 0; i < individuals.length; i++) {
            out.format("%1$" + maxEntryWidth + "d", i + 1);
        }
        out.println();
        printRepeat(out, maxHeaderLength + 1 + maxRankWidth + 1, " ");
        printRepeat(out, individuals.length * maxEntryWidth, "_");
        out.println();
        for (int i = 0; i < individuals.length; i++) {
            final String sname = individuals[ranks.get(i)].sname;
            out.format("%1$" + maxHeaderLength + "s %2$" + maxRankWidth + "d|",
                    sname, i + 1);
            for (int j = 0; j < individuals.length; j++) {
                int x = matrix[ranks.get(i)][ranks.get(j)];
                if (i == j) {
                    printRepeat(out, maxEntryWidth - 1, " ");
                    out.print("X");
                } else if (x != 0) {
                    out.format("%1$" + maxEntryWidth + "d", x);
                } else {
                    printRepeat(out, maxEntryWidth - 1, " ");
                    out.print(".");
                }
            }
            out.println();
        }
        out.println();
        return;
    }

    public void dumpToXML(File file) throws DumpToXMLException {
        try {
            DocumentBuilder builder = _factory.newDocumentBuilder();
            Document document = builder.newDocument();
            Element rankingE = document.createElement("ranking");
            document.appendChild(rankingE);
            rankingE.appendChild(rankingId.toXML(document));
            rankingE.appendChild(interactionSpec.toXML(document));
            Element individualsE = document.createElement("individuals");
            individualsE.setAttribute("n", String.valueOf(individuals.length));
            rankingE.appendChild(individualsE);
            for (int i = 0; i < individuals.length; i++) {
                individualsE.appendChild(individuals[i].toXML(document));
            }
            Element ranksE = document.createElement("ranks");
            ranksE.setAttribute("n", String.valueOf(ranks.size()));
            rankingE.appendChild(ranksE);
            for (int rank = 0; rank < ranks.size(); rank++) {
                Element e = document.createElement("rank");
                e.setAttribute("i", String.valueOf(ranks.get(rank)));
                ranksE.appendChild(e);
            }
            Element matrixE = document.createElement("matrix");
            matrixE.setAttribute("n", String.valueOf(individuals.length));
            rankingE.appendChild(matrixE);
            for (int i = 0; i < individuals.length; i++) {
                for (int j = 0; j < individuals.length; j++) {
                    if (matrix[i][j] == 0)
                        continue;
                    Element e = document.createElement("entry");
                    e.setAttribute("i", String.valueOf(i));
                    e.setAttribute("j", String.valueOf(j));
                    e.setAttribute("v", String.valueOf(matrix[i][j]));
                    matrixE.appendChild(e);
                }
            }
            TransformerFactory.newInstance().newTransformer().transform(
                    new DOMSource(document), new StreamResult(file));
        } catch (ParserConfigurationException e) {
            throw new DumpToXMLException("XML parser configuration error", e);
        } catch (TransformerConfigurationException e) {
            throw new DumpToXMLException("XML transformer configuration error",
                    e);
        } catch (TransformerException e) {
            throw new DumpToXMLException("XML transformer error", e);
        } catch (TransformerFactoryConfigurationError e) {
            throw new DumpToXMLException(
                    "XML transformer factory configuration error", e);
        }
        return;
    }

    public static Cage loadFromXML(File file) throws LoadFromXMLException {
        try {
            DocumentBuilder builder = _factory.newDocumentBuilder();
            Document document = builder.parse(file);
            Element rankingE = document.getDocumentElement();
            RankingIdentifier rankingId = null;
            InteractionSpec interactionSpec = null;
            Individual[] individuals = null;
            int[] ranks = null;
            int[][] matrix = null;
            for (Node child = rankingE.getFirstChild(); child != null; child = child
                    .getNextSibling()) {
                if (child.getNodeType() != Node.ELEMENT_NODE)
                    continue;
                Element e = (Element) child;
                if (e.getNodeName().equals("rankingId")) {
                    rankingId = RankingIdentifier.fromXML(e);
                } else if (e.getNodeName().equals("interactionSpec")) {
                    interactionSpec = InteractionSpec.fromXML(e);
                } else if (e.getNodeName().equals("individuals")) {
                    int n = Integer.parseInt(e.getAttribute("n"));
                    individuals = new Individual[n];
                    int count = 0;
                    for (Element ec = (Element) e.getFirstChild(); ec != null; ec = (Element) ec
                            .getNextSibling()) {
                        if (count > n)
                            throw new LoadFromXMLException(
                                    "element <individuals> contains too many <individual> subelements");
                        individuals[count] = Individual.fromXML(ec);
                        count++;
                    }
                    if (count < n)
                        throw new LoadFromXMLException(
                                "element <individuals> contains too few <individual> subelements");
                } else if (e.getNodeName().equals("ranks")) {
                    int n = Integer.parseInt(e.getAttribute("n"));
                    ranks = new int[n];
                    int count = 0;
                    for (Element ec = (Element) e.getFirstChild(); ec != null; ec = (Element) ec
                            .getNextSibling()) {
                        if (count > n)
                            throw new LoadFromXMLException(
                                    "element <ranks> contains too many <rank> subelements");
                        ranks[count] = Integer.parseInt(ec.getAttribute("i"));
                        count++;
                    }
                    if (count < n)
                        throw new LoadFromXMLException(
                                "element <ranks> contains too few <rank> subelements");
                } else if (e.getNodeName().equals("matrix")) {
                    int n = Integer.parseInt(e.getAttribute("n"));
                    matrix = new int[n][n];
                    for (int i = 0; i < n; i++)
                        for (int j = 0; j < n; j++)
                            matrix[i][j] = 0;
                    for (Element ec = (Element) e.getFirstChild(); ec != null; ec = (Element) ec
                            .getNextSibling()) {
                        int i = Integer.parseInt(ec.getAttribute("i"));
                        int j = Integer.parseInt(ec.getAttribute("j"));
                        int v = Integer.parseInt(ec.getAttribute("v"));
                        if (i < 0 || i >= n)
                            throw new LoadFromXMLException(
                                    "index i of element <entry i='" + i
                                            + "' .../> is out of bounds");
                        if (j < 0 || j >= n)
                            throw new LoadFromXMLException(
                                    "index j of element <entry ... j='" + j
                                            + "' .../> is out of bounds");
                        if (v <= 0)
                            throw new LoadFromXMLException(
                                    "value v of element <entry ... v='" + v
                                            + "/> element is not positive");
                        matrix[i][j] = v;
                    }
                }
            }
            return new Cage(rankingId, interactionSpec, individuals, ranks,
                    matrix);
        } catch (ParserConfigurationException e) {
            throw new LoadFromXMLException("XML parser configuration error", e);
        } catch (SAXException e) {
            throw new LoadFromXMLException("XML parsing error", e);
        } catch (IOException e) {
            throw new LoadFromXMLException("error reading file " + file, e);
        }
    }

}