package babase.ranker;

import info.clearthought.layout.TableLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.PrintStream;

import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;

import babase.ranker.ranking.Ranking;
import babase.ranker.ranking.RankingChangeEvent;
import babase.ranker.ranking.RankingChangedListener;

public class HistoryPanel extends JPanel implements RankingChangedListener,
        ActionListener {

    /**
     * Compiler-generated <code>serialVersionUID</code>.
     */
    private static final long serialVersionUID = 1997859215699352960L;

    private final Ranking _ranks;

    private final DefaultListModel _undoListModel;

    private final DefaultListModel _redoListModel;

    private final JList _undoJList;

    private final JList _redoJList;

    private PrintStream _out;

    public HistoryPanel(Ranking ranks) {
        super();
        _ranks = ranks;
        _undoListModel = new DefaultListModel();
        _redoListModel = new DefaultListModel();
        _undoJList = new JList(_undoListModel);
        _redoJList = new JList(_redoListModel);
        _out = System.out;

        _ranks.addChangedListener(this);

        double border = 5;
        double[] tableLayoutWidths = { border, TableLayout.FILL, border,
                TableLayout.FILL, border };
        double[] tableLayoutHeights = { border, TableLayout.PREFERRED, border,
                TableLayout.FILL, border };
        setLayout(new TableLayout(tableLayoutWidths, tableLayoutHeights));
        JButton undoButton = new JButton("Undo");
        undoButton.setActionCommand("Undo");
        undoButton.addActionListener(this);
        add(undoButton, "1 1 r c");

        JButton redoButton = new JButton("Redo");
        redoButton.setActionCommand("Redo");
        redoButton.addActionListener(this);
        add(redoButton, "3 1 l c");

        _undoJList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        JScrollPane undoScroll = new JScrollPane(_undoJList,
                JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
                JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        add(undoScroll, "1 3 f f");

        _undoJList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        JScrollPane redoScroll = new JScrollPane(_redoJList,
                JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
                JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        add(redoScroll, "3 3 f f");

        return;
    }

    public void setOut(PrintStream out) {
        _out = out;
        return;
    }

    void redo() {
        if (_redoListModel.size() == 0) {
            _out.println("Nothing to redo!");
            return;
        }
        RankingChangeEvent e = (RankingChangeEvent) _redoListModel.remove(0);
        if (e instanceof RankingChangeEvent.Slide) {
            RankingChangeEvent.Slide slideEvent = (RankingChangeEvent.Slide) e;
            _ranks.slide(slideEvent.from, slideEvent.to, this);
        } else if (e instanceof RankingChangeEvent.Swap) {
            RankingChangeEvent.Swap swapEvent = (RankingChangeEvent.Swap) e;
            _ranks.swap(swapEvent.i, swapEvent.j, this);
        } else if (e instanceof RankingChangeEvent.Permute) {
            RankingChangeEvent.Permute permuteEvent = (RankingChangeEvent.Permute) e;
            _ranks.permute(permuteEvent.after, this);
        }
        _undoListModel.addElement(e);
        _undoJList.ensureIndexIsVisible(_undoListModel.size() - 1);
        return;
    }

    void undo() {
        if (_undoListModel.size() == 0) {
            _out.println("Nothing to undo!");
            return;
        }
        RankingChangeEvent e = (RankingChangeEvent) _undoListModel
                .remove(_undoListModel.size() - 1);
        if (e instanceof RankingChangeEvent.Slide) {
            RankingChangeEvent.Slide slideEvent = (RankingChangeEvent.Slide) e;
            _ranks.slide(slideEvent.to, slideEvent.from, this);
        } else if (e instanceof RankingChangeEvent.Swap) {
            RankingChangeEvent.Swap swapEvent = (RankingChangeEvent.Swap) e;
            _ranks.swap(swapEvent.j, swapEvent.i, this);
        } else if (e instanceof RankingChangeEvent.Permute) {
            RankingChangeEvent.Permute permuteEvent = (RankingChangeEvent.Permute) e;
            _ranks.permute(permuteEvent.before, this);
        }
        _redoListModel.add(0, e);
        _redoJList.ensureIndexIsVisible(0);
        return;
    }

    public void rankingChanged(RankingChangeEvent e) {
        if (e.source == this)
            return;
        _undoListModel.addElement(e);
        _undoJList.ensureIndexIsVisible(_undoListModel.size() - 1);
        if (_redoListModel.size() != 0) {
            _redoListModel.clear();
        }
        return;
    }

    public void actionPerformed(ActionEvent e) {
        if (e.getActionCommand().equals("Undo")) {
            int selected = _undoJList.getSelectedIndex();
            if (selected == -1) {
                undo();
            } else if (selected == _undoListModel.size() - 1) {
                _undoJList.clearSelection();
                undo();
            } else {
                _out.print("Undoing " + (_undoListModel.size() - selected - 1)
                        + " operation(s)... ");
                for (int i = _undoListModel.size() - 1; i > selected; i--) {
                    undo();
                }
                _out.println("Done!");
            }
        } else if (e.getActionCommand().equals("Redo")) {
            int selected = _redoJList.getSelectedIndex();
            if (selected == -1) {
                redo();
            } else if (selected == 0) {
                _redoJList.clearSelection();
                redo();
            } else {
                _out.print("Redoing " + selected + " operation(s)... ");
                for (int i = 0; i < selected; i++) {
                    redo();
                }
                _out.println("Done!");
            }
        }
        return;
    }
}
