package babase.ranker;

import java.awt.Color;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Insets;

import javax.swing.ButtonGroup;
import javax.swing.JPanel;
import javax.swing.JToggleButton;
import javax.swing.UIManager;

import babase.ranker.ranking.Ranking;
import babase.ranker.ranking.RankingChangeEvent;
import babase.ranker.ranking.RankingChangedListener;

/**
 * Implements the display of the interaction matrix according to the current
 * ranking.
 * 
 * @author junyang
 * 
 */
public class InteractionMatrixDisplay extends JPanel implements
        RankingChangedListener {

    /**
     * Compiler-generated <code>serialVersionUID</code>.
     */
    private static final long serialVersionUID = 3799732940123199825L;

    /**
     * Implements the display of a particular entry in the interaction matrix.
     * 
     * @author junyang
     * 
     */
    public static class DisplayEntry extends JToggleButton {

        /**
         * Compiler-generated <code>serialVersionUID</code>.
         */
        private static final long serialVersionUID = 997196787831598016L;

        /**
         * Margin between displayed text and its border.
         */
        private static final Insets insets = new Insets(0, 0, 0, 0);

        /**
         * @param i
         *            Row index (<code>0</code>-based) of the entry.
         * @param j
         *            Column index (<code>0</code>-based) of the entry.
         */
        public DisplayEntry(int i, int j) {
            super();
            setFont((Font) UIManager.get(getClass().getName() + ".font"));
            setMargin(insets);
            if (i == j)
                setBackground(Color.LIGHT_GRAY);
            else
                setBackground(Color.WHITE);
            return;
        }
        
        public void updateUI() {
            super.updateUI();
            setFont((Font) UIManager.get(getClass().getName() + ".font"));
            return;
        }

        /**
         * Set the value to display in this entry.
         * 
         * @param value
         */
        public void setValue(int value) {
            setText((value == 0) ? "" : Integer.toString(value));
            if (value == 0) {
                setToolTipText(null);
            } else {
                setToolTipText(Integer.toString(value));
            }
            return;
        }

    }

    /**
     * The ranking. This <code>InteractionMatrixDisplay</code> object will not
     * alter this ranking, but will listen to any changes made to it and adjust
     * the display of the interaction matrix accordingly.
     */
    private final Ranking _ranks;

    /**
     * The interaction matrix. The content of this matrix will not be altered by
     * this InteractionMatrixDisplay object.
     */
    private final int[][] _matrix;

    /**
     * A 2D array of Swing objects that display the entries of the interaction
     * matrix according the current ranking.
     */
    private final DisplayEntry[][] _displayEntries;

    /**
     * Used to ensure that at most one of the display entries can be selected.
     */
    private final ButtonGroup _displayGroup;

    /**
     * An invisible button in {@link #_displayGroup} which, when selected,
     * unselects all display entries.
     */
    private final JToggleButton _displayNoneSelected;
    
    public InteractionMatrixDisplay(Ranking ranks, int[][] matrix) {
        super();
        _ranks = ranks;
        _matrix = matrix;
        _displayEntries = new DisplayEntry[_ranks.size()][_ranks.size()];
        _displayGroup = new ButtonGroup();
        _displayNoneSelected = new JToggleButton();

        _ranks.addChangedListener(this);
        setLayout(new GridLayout(_ranks.size(), _ranks.size()));
        for (int i = 0; i < _ranks.size(); i++) {
            for (int j = 0; j < _ranks.size(); j++) {
                _displayEntries[i][j] = new DisplayEntry(i, j);
                add(_displayEntries[i][j]);
                _displayGroup.add(_displayEntries[i][j]);
            }
        }
        // Add the invisible button, but do not display it:
        _displayGroup.add(_displayNoneSelected);
        refresh();
        return;
    }
    
    public Ranking getRanks() {
        return _ranks;
    }

    public int getDisplayCellWidth() {
        int maxDisplayLength = 0;
        for (int i = 0; i < _matrix.length; i++)
            for (int j = 0; j < _matrix[i].length; j++)
                if (Integer.toString(_matrix[i][j]).length() > maxDisplayLength)
                    maxDisplayLength = Integer.toString(_matrix[i][j]).length();
        return maxDisplayLength;
    }

    public int getDisplayCellHeight() {
        return 1;
    }

    /**
     * Select the interaction matrix entry currently displayed at position (<code>i</code>,
     * <code>j</code>), which corresponds to the number of interactions with
     * the <code>i</code>-th (0-based) ranked individual as the actor and
     * <code>j</code>-th ranked individual as the actee.
     * 
     * @param i
     * @param j
     */
    public void selectDisplayEntry(int i, int j) {
        _displayEntries[i][j].setSelected(true);
        return;
    }

    /**
     * Update the display at position (<code>i</code>, <code>j</code>),
     * usually in response to a ranking change. The entry displayed at this
     * position corresponds to the number of interactions with the
     * <code>i</code>-th (<code>0</code>-based) ranked individual as the
     * actor and <code>j</code>-th ranked individual as the actee.
     * 
     * If position (<code>i</code>, <code>j</code>) is previously
     * selected, it will be unselected, because this refresh may be responding
     * to a change involving the selected entry.
     * 
     * @param i
     * @param j
     */
    private void refreshDisplayEntry(int i, int j) {
        if (_displayEntries[i][j].isSelected())
            _displayNoneSelected.setSelected(true);
        _displayEntries[i][j].setValue(_matrix[_ranks.get(i)][_ranks.get(j)]);
        return;
    }

    /**
     * Refresh the entire display.
     */
    public void refresh() {
        for (int i = 0; i < _ranks.size(); i++)
            for (int j = 0; j < _ranks.size(); j++)
                refreshDisplayEntry(i, j);
        return;
    }

    /*
     * (non-Javadoc)
     * 
     * @see babase.ranker.ranking.RankingChangedListener#rankingChanged(babase.ranker.ranking.RankingChangeEvent)
     */
    public void rankingChanged(RankingChangeEvent e) {
        if (e instanceof RankingChangeEvent.Slide) {
            // Selectively refresh display:
            RankingChangeEvent.Slide slideEvent = (RankingChangeEvent.Slide) e;
            int lower = (slideEvent.from > slideEvent.to) ? slideEvent.to
                    : slideEvent.from;
            int upper = (slideEvent.from > slideEvent.to) ? slideEvent.from
                    : slideEvent.to;
            for (int i = 0; i < _ranks.size(); i++) {
                for (int j = lower; j <= upper; j++) {
                    refreshDisplayEntry(i, j);
                }
            }
            for (int i = lower; i <= upper; i++) {
                for (int j = 0; j < lower; j++) {
                    refreshDisplayEntry(i, j);
                }
                for (int j = upper + 1; j < _ranks.size(); j++) {
                    refreshDisplayEntry(i, j);
                }
            }
        } else if (e instanceof RankingChangeEvent.Swap) {
            int i = ((RankingChangeEvent.Swap) e).i;
            int j = ((RankingChangeEvent.Swap) e).j;
            // Selectively refresh display:
            for (int k = 0; k < _ranks.size(); k++) {
                refreshDisplayEntry(i, k);
                refreshDisplayEntry(j, k);
            }
            for (int k = 0; k < _ranks.size(); k++) {
                if (k == i)
                    continue;
                if (k == j)
                    continue;
                refreshDisplayEntry(k, i);
                refreshDisplayEntry(k, j);
            }
        } else if (e instanceof RankingChangeEvent.Permute) {
            refresh();
        }
        return;
    }

}
