package babase.ranker;

import info.clearthought.layout.TableLayout;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.io.PrintStream;
import java.util.ArrayList;

import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import babase.db.Individual;
import babase.ranker.ranking.Ranking;
import babase.ranker.ranking.RankingChangeEvent;
import babase.ranker.ranking.RankingChangedListener;

public class StatsPanel extends JPanel implements RankingChangedListener,
        ActionListener {

    /**
     * 
     */
    private static final long serialVersionUID = 1330200575513632020L;

    public static class Stats {

        /**
         * Number of positive entries below diagonal.
         */
        protected int numPairsReversed;

        /**
         * Number of entries below diagonal greater than <code>1</code>.
         */
        protected int numPairsSignificantlyReversed;

        /**
         * Maximum value among positive entries below diagonal (<code>0</code>
         * if there are no positive entries below diagonal).
         */
        protected int maxReversalMagnitude;

        protected ArrayList<Integer> iWithMaxReversalMagnitude;

        protected ArrayList<Integer> jWithMaxReversalMagnitude;

        /**
         * Maximum difference in ranks (i.e., <code>j-i</code>) among
         * posititve entries below diagonal (<code>0</code> if there are no
         * positive entries below diagonal):
         */
        protected int maxReversalDistance;

        protected ArrayList<Integer> iWithMaxReversalDistance;

        protected ArrayList<Integer> jWithMaxReversalDistance;

        public Stats(Ranking ranks, int[][] matrix) {
            compute(ranks, matrix);
            return;
        }

        public void compute(Ranking ranks, int[][] matrix) {
            numPairsReversed = 0;
            numPairsSignificantlyReversed = 0;
            maxReversalMagnitude = 0;
            maxReversalDistance = 0;
            for (int i = 0; i < ranks.size(); i++) {
                for (int j = 0; j < i; j++) {
                    if (matrix[ranks.get(i)][ranks.get(j)] <= 0)
                        continue;
                    numPairsReversed++;
                    if (matrix[ranks.get(i)][ranks.get(j)] > 1)
                        numPairsSignificantlyReversed++;
                    if (matrix[ranks.get(i)][ranks.get(j)] > maxReversalMagnitude)
                        maxReversalMagnitude = matrix[ranks.get(i)][ranks
                                .get(j)];
                    if (i - j > maxReversalDistance)
                        maxReversalDistance = i - j;
                }
            }
            iWithMaxReversalMagnitude = new ArrayList<Integer>();
            jWithMaxReversalMagnitude = new ArrayList<Integer>();
            iWithMaxReversalDistance = new ArrayList<Integer>();
            jWithMaxReversalDistance = new ArrayList<Integer>();
            if (numPairsReversed > 0) {
                for (int i = 0; i < ranks.size(); i++) {
                    for (int j = 0; j < i; j++) {
                        if (matrix[ranks.get(i)][ranks.get(j)] == maxReversalMagnitude) {
                            iWithMaxReversalMagnitude.add(i);
                            jWithMaxReversalMagnitude.add(j);
                        }
                        if (matrix[ranks.get(i)][ranks.get(j)] > 0
                                && i - j == maxReversalDistance) {
                            iWithMaxReversalDistance.add(i);
                            jWithMaxReversalDistance.add(j);
                        }
                    }
                }
            }
            return;
        }
    }

    protected final InteractionMatrixDisplay _interactionMatrixDisplay;

    protected final Ranking _ranks;

    protected final int[][] _matrix;

    protected final Individual[] _individuals;

    protected Stats _stats;

    protected boolean _statsUpToDate;

    protected final String[] _componentNames = { "# pairs rev.:",
            "# pairs significantly rev.:", "Max. rev. magnitude:",
            "Pairs with max. rev. magnitude:", "Max. rev. distance:",
            "Pairs with max. rev. distance:" };

    protected final String[] _componentDescriptions = {
            "<html>" + "<b>Number of reversed pairs</b>:<br>"
                    + "i.e., number of positive entries below diagonal"
                    + "</html>",
            "<html>" + "<b>Number of significantly reversed pairs</b>:<br>"
                    + "i.e., number of positive entries below diagonal<br>"
                    + "with at least 2 interactions" + "</html>",
            "<html>"
                    + "<b>Maximum reversal magnitude</b>:<br>"
                    + "i.e., the maximum value among positive entries below diagonal,<br>"
                    + "or 0 if there are no positive entries below diagonal"
                    + "</html>",
            "<html>"
                    + "<b>Pairs with the maximum reversal magnitude</b>:<br>"
                    + "select the pair to highlight the corresponding matrix entry"
                    + "</html>",
            "<html>"
                    + "<b>Maximum reversal distance in ranks</b>:<br>"
                    + "i.e., the maximum difference between actor/actee ranks<br>"
                    + "among posititve entries below diagonal,<br>"
                    + "or 0 if there are no positive entries below diagonal"
                    + "</html>",
            "<html>"
                    + "<b>Pairs with the maximum reversal distance</b>:<br>"
                    + "select the pair to highlight the corresponding matrix entry"
                    + "</html>" };

    protected final Component[] _components;

    protected final JLabel _numPairsReversed;

    protected final JLabel _numPairsSignificantlyReversed;

    protected final JLabel _maxReversalMagnitude;

    protected final JComboBox _pairsWithMaxReversalMagnitude;

    protected final JLabel _maxReversalDistance;

    protected final JComboBox _pairsWithMaxReversalDistance;

    public StatsPanel(InteractionMatrixDisplay interactionMatrixDisplay,
            Ranking ranks, Individual[] individuals, int[][] matrix) {
        super();
        _interactionMatrixDisplay = interactionMatrixDisplay;
        _ranks = ranks;
        _matrix = matrix;
        _individuals = individuals;
        _stats = new Stats(ranks, matrix);
        _statsUpToDate = false;
        _numPairsReversed = new JLabel();
        _numPairsReversed.setHorizontalAlignment(JLabel.LEFT);
        _numPairsSignificantlyReversed = new JLabel();
        _numPairsSignificantlyReversed.setHorizontalAlignment(JLabel.LEFT);
        _maxReversalMagnitude = new JLabel();
        _maxReversalMagnitude.setHorizontalAlignment(JLabel.LEFT);
        _pairsWithMaxReversalMagnitude = new JComboBox();
        _maxReversalDistance = new JLabel();
        _maxReversalDistance.setHorizontalAlignment(JLabel.LEFT);
        _pairsWithMaxReversalDistance = new JComboBox();

        _ranks.addChangedListener(this);

        double border = 5;
        double[] tableLayoutWidths = { border, TableLayout.PREFERRED, border,
                TableLayout.PREFERRED, border };
        double[] tableLayoutHeights = { border, TableLayout.PREFERRED, border,
                TableLayout.PREFERRED, border, TableLayout.PREFERRED, border,
                TableLayout.PREFERRED, border, TableLayout.PREFERRED, border,
                TableLayout.PREFERRED, border };
        this.setLayout(new TableLayout(tableLayoutWidths, tableLayoutHeights));
        _components = new Component[] { _numPairsReversed,
                _numPairsSignificantlyReversed, _maxReversalMagnitude,
                _pairsWithMaxReversalMagnitude, _maxReversalDistance,
                _pairsWithMaxReversalDistance };
        for (int i = 0; i < _components.length; i++) {
            final JLabel label = new JLabel(_componentNames[i]);
            label.setToolTipText(_componentDescriptions[i]);
            add(label, "" + 1 + " " + (1 + i * 2) + " r c");
            add(_components[i], "" + 3 + " " + (1 + i * 2) + " f c");
        }
        addComponentListener(new ComponentAdapter() {
            public void componentShown(ComponentEvent e) {
                if (!_statsUpToDate) {
                    redisplay();
                }
                return;
            }
        });
        return;
    }

    protected static Object makeObject(final String s) {
        return new Object() {
            public String toString() {
                return s;
            }
        };
    }

    protected void redisplay() {
        if (!_statsUpToDate) {
            _stats.compute(_ranks, _matrix);
            _statsUpToDate = true;
        }
        _numPairsReversed.setText(Integer.toString(_stats.numPairsReversed));
        _numPairsSignificantlyReversed.setText(Integer
                .toString(_stats.numPairsSignificantlyReversed));
        _maxReversalMagnitude.setText(Integer
                .toString(_stats.maxReversalMagnitude));
        _pairsWithMaxReversalMagnitude.removeActionListener(this);
        _pairsWithMaxReversalMagnitude.removeAllItems();
        _pairsWithMaxReversalMagnitude
                .addItem(makeObject((_stats.iWithMaxReversalMagnitude.isEmpty()) ? "(None)"
                        : "(Click to Choose)"));
        for (int k = 0; k < _stats.iWithMaxReversalMagnitude.size(); k++) {
            int i = _stats.iWithMaxReversalMagnitude.get(k);
            int j = _stats.jWithMaxReversalMagnitude.get(k);
            _pairsWithMaxReversalMagnitude
                    .addItem(makeObject(_individuals[_ranks.get(i)].sname
                            + " (" + (i + 1) + ") vs. "
                            + _individuals[_ranks.get(j)].sname + " ("
                            + (j + 1) + ")"));
        }
        _pairsWithMaxReversalMagnitude.addActionListener(this);
        _maxReversalDistance.setText(Integer
                .toString(_stats.maxReversalDistance));
        _pairsWithMaxReversalDistance.removeActionListener(this);
        _pairsWithMaxReversalDistance.removeAllItems();
        _pairsWithMaxReversalDistance
                .addItem(makeObject((_stats.iWithMaxReversalDistance.isEmpty()) ? "(None)"
                        : "(Click to Choose)"));
        for (int k = 0; k < _stats.iWithMaxReversalDistance.size(); k++) {
            int i = _stats.iWithMaxReversalDistance.get(k);
            int j = _stats.jWithMaxReversalDistance.get(k);
            _pairsWithMaxReversalDistance
                    .addItem(makeObject(_individuals[_ranks.get(i)].sname
                            + " (" + (i + 1) + ") vs. "
                            + _individuals[_ranks.get(j)].sname + " ("
                            + (j + 1) + ")"));
        }
        _pairsWithMaxReversalDistance.addActionListener(this);
        return;
    }

    public void rankingChanged(RankingChangeEvent e) {
        _statsUpToDate = false;
        if (isVisible())
            redisplay();
        return;
    }

    public void actionPerformed(ActionEvent e) {
        JComboBox source = (JComboBox) e.getSource();
        int selectedIndex = source.getSelectedIndex();
        if (selectedIndex == -1 || selectedIndex == 0)
            return;
        if (source == _pairsWithMaxReversalMagnitude) {
            int i = _stats.iWithMaxReversalMagnitude.get(selectedIndex - 1);
            int j = _stats.jWithMaxReversalMagnitude.get(selectedIndex - 1);
            _interactionMatrixDisplay.selectDisplayEntry(i, j);
        } else if (source == _pairsWithMaxReversalDistance) {
            int i = _stats.iWithMaxReversalDistance.get(selectedIndex - 1);
            int j = _stats.jWithMaxReversalDistance.get(selectedIndex - 1);
            _interactionMatrixDisplay.selectDisplayEntry(i, j);
        }
        return;
    }

    public void print(PrintStream out) {
        int maxWidth = 0;
        for (int i = 0; i < _componentNames.length; i++) {
            int len = _componentNames[i].length();
            if (maxWidth < len)
                maxWidth = len;
        }
        for (int i = 0; i < _components.length; i++) {
            out.format("%1$" + maxWidth + "s ", _componentNames[i]);
            if (_components[i] instanceof JLabel) {
                out.print(((JLabel) _components[i]).getText());
            } else if (_components[i] instanceof JComboBox) {
                JComboBox box = (JComboBox) _components[i];
                for (int j = 1; j < box.getItemCount(); j++) {
                    if (j != 1)
                        out.print(", ");
                    out.print(box.getItemAt(j));
                }
            }
            out.println();
        }
        return;
    }
}
