package babase.ranker.ranking;

import java.util.ArrayList;

/**
 * A permutation of integers from <code>0</code> to <code>n-1</code> that
 * represents a linear ranking of <code>n</code> individuals. Each individual
 * is identified by an integer from <code>0</code> to <code>n-1</code>.
 * 
 * @author junyang
 * 
 */
public class Ranking {

    /**
     * A permutation of integers from <code>0</code> to <code>n-1</code>
     * that represents a linear ranking of <code>n</code> individuals.
     * Specifically, the individual identified by <code>_ranks[i]</code> is
     * ranked at <code>i</code>.
     */
    protected int[] _ranks;

    /**
     * A list of {@link RankingChangedListener}s that will be notified
     * immediately after this <code>Ranking</code> object changes.
     */
    protected ArrayList<RankingChangedListener> _changedListeners;

    /**
     * Construct an initial ranking of <code>n</code> individuals in their
     * default order <code>0</code> to <code>n-1</code>.
     * 
     * @param n
     */
    public Ranking(int n) {
        _ranks = new int[n];
        for (int i = 0; i < n; i++) {
            _ranks[i] = i;
        }
        _changedListeners = new ArrayList<RankingChangedListener>();
        return;
    }

    /**
     * Construct an initial ranking of <code>n</code> individuals in the order
     * specified by <code>ranks</code>.
     * 
     * @param ranks
     *            A permutation of integers from <code>0</code> to
     *            <code>n-1</code> that represents a linear ranking of
     *            <code>n</code> individuals. Specifically, the individual
     *            identified by <code>_ranks[i]</code> is ranked at
     *            <code>i</code>.
     */
    public Ranking(int[] ranks) {
        this(ranks.length);
        boolean[] seen = new boolean[ranks.length];
        for (int i = 0; i < seen.length; i++)
            seen[i] = false;
        for (int i = 0; i < ranks.length; i++) {
            _ranks[i] = ranks[i];
            assert (seen[_ranks[i]] == false);
            seen[_ranks[i]] = true;
        }
        return;
    }

    /**
     * @param listener
     *            A {@link RankingChangedListener} object to be added. It will
     *            be notified immediately after each change to this
     *            <code>Ranking</code>.
     */
    public void addChangedListener(RankingChangedListener listener) {
        _changedListeners.add(listener);
        return;
    }

    /**
     * @param listener
     *            A {@link RankingChangedListener} object to be removed.
     */
    public void removeChangedListener(RankingChangedListener listener) {
        _changedListeners.remove(listener);
        return;
    }

    /**
     * @return The number of individuals in the ranking.
     */
    public int size() {
        return _ranks.length;
    }

    /**
     * @param i
     *            A <code>0</code>-based rank.
     * @return The <code>i</code>-th ranked individual, identified by an
     *         integer between <code>0</code> and {@link #size()}<code>-1</code>.
     */
    public int get(int i) {
        return _ranks[i];
    }

    /**
     * @return The entire ranking as an array of integers. The <code>i</code>-th
     *         array element is an an integer between <code>0</code> and
     *         {@link #size()}<code>-1</code> that identifies the
     *         <code>i</code>-th ranked individual.
     */
    public int[] get() {
        return _ranks.clone();
    }

    /**
     * Swap the ranks of individuals currently ranked at <code>i</code> and
     * <code>j</code> (both <code>0</code>-based ranks). This method
     * generates a {@link RankingChangeEvent.Swap} event.
     * 
     * @param i
     * @param j
     * @param swapper
     *            The object calling this method.
     */
    public void swap(int i, int j, Object swapper) {
        if (i == j)
            return;
        int tmp = _ranks[i];
        _ranks[i] = _ranks[j];
        _ranks[j] = tmp;
        // Copying the list of listeners is necessary because listerners may
        // remove themselves when rankingChanged is called:
        for (RankingChangedListener listener : _changedListeners
                .toArray(new RankingChangedListener[_changedListeners.size()])) {
            listener.rankingChanged(new RankingChangeEvent.Swap(swapper, this,
                    i, j));
        }
        return;
    }

    /**
     * Slide an individual to a new rank. The result of this operation is that
     * this individual attains the new rank, while the individuals between the
     * old and new ranks are shifted by one towards the old rank of the
     * individual. This method generates a {@link RankingChangeEvent.Slide}
     * event.
     * 
     * @param from
     *            The current rank (<code>0</code>-based) of the individual.
     * @param to
     *            The new rank (<code>0</code>-based) to slide to.
     * @param slider
     *            The object calling this method.
     */
    public void slide(int from, int to, Object slider) {
        if (from == to)
            return;
        int tmp = _ranks[from];
        if (from > to) {
            for (int i = from; i > to; i--) {
                _ranks[i] = _ranks[i - 1];
            }
        } else {
            for (int i = from; i < to; i++) {
                _ranks[i] = _ranks[i + 1];
            }
        }
        _ranks[to] = tmp;
        // Copying the list of listeners is necessary because listeners may
        // remove themselves when rankingChanged is called:
        for (RankingChangedListener listener : _changedListeners
                .toArray(new RankingChangedListener[_changedListeners.size()])) {
            listener.rankingChanged(new RankingChangeEvent.Slide(slider, this,
                    from, to));
        }
        return;
    }

    /**
     * Permute the ranking. This method generates a
     * {@link RankingChangeEvent.Permute} event.
     * 
     * @param newRanks
     *            The new ranking, a permutation of integers between
     *            <code>0</code> and {@link #size()}<code>-1</code>.
     * @param permuter
     *            The object calling this method.
     */
    public void permute(int[] newRanks, Object permuter) {
        int[] before = _ranks.clone();
        for (int i = 0; i < _ranks.length; i++) {
            _ranks[i] = newRanks[i];
        }
        // Copying the list of listeners is necessary because listeners may
        // remove themselves when rankingChanged is called:
        for (RankingChangedListener listener : _changedListeners
                .toArray(new RankingChangedListener[_changedListeners.size()])) {
            listener.rankingChanged(new RankingChangeEvent.Permute(permuter,
                    this, before, newRanks));
        }
        return;
    }
}
